import logging
import os
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
import sandbox.projects.resource_types as resource_types
import sandbox.projects.images.resource_types as images_resource_types


class ImagesComposeRobotBundle(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        package_to_bundle_files = sdk2.parameters.Dict("Package to bundle paths",
                                                       required=True)
        package = sdk2.parameters.Resource("Package",
                                           multiple=False,
                                           resource_type=images_resource_types.IMAGES_CM_PACKAGE,
                                           required=True)
        sw_config = sdk2.parameters.Resource("Shard writer config",
                                             multiple=False,
                                             resource_type=resource_types.IMAGES_SHARDWRITER_CONFIG,
                                             required=True)
        resource_name = sdk2.parameters.String("Bundle type", required=False)
        bundle = sdk2.parameters.ParentResource("Deploy bundle", required=False)

    class Requirements(sdk2.Requirements):
        disk_space = 30 * 1024

    def on_execute(self):
        self.__setup_bundle()
        self.__unpack_package()
        self.__copy_package_to_bundle()
        self.__copy_config_to_bundle()
        self.bundle_data.ready()

    def __setup_bundle(self):
        bundle = None
        if self.Parameters.bundle:
            bundle = self.Parameters.bundle
        else:
            resource_name = self.Parameters.resource_name
            bundle_resource_type = getattr(resource_types, resource_name)
            bundle = bundle_resource_type(self, "Deploy bundle", "bundle")
        self.bundle_data = sdk2.ResourceData(bundle)
        self.bundle_data.path.mkdir(mode=0o755, parents=True, exist_ok=True)

    def __unpack_package(self):
        package = self.Parameters.package
        package_data = sdk2.ResourceData(package)
        logging.info("Package path {}".format(package_data.path))
        task_dir = os.getcwd()
        self.package_dir = os.path.join(task_dir, "package")
        sdk2.path.Path(self.package_dir).mkdir(mode=0o755, exist_ok=True)
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("tar")) as pl:
            sp.Popen("tar xf {} -C {}".format(package_data.path, self.package_dir),
                     shell=True,
                     stdout=pl.stdout,
                     stderr=sp.STDOUT).wait()

    def __copy_package_to_bundle(self):
        package_to_bundle = self.Parameters.package_to_bundle_files
        for package_path, bundle_path in package_to_bundle.iteritems():
            logging.info("Package file {} to bundle file {}".format(package_path, bundle_path))
            sdk2.paths.copy_path(os.path.join(self.package_dir, package_path),
                                 str(self.bundle_data.path.joinpath(bundle_path)))

    def __copy_config_to_bundle(self):
        sw_config = self.Parameters.sw_config
        sw_config_data = sdk2.ResourceData(sw_config)
        logging.info("SW config path {}".format(sw_config_data.path))
        sdk2.paths.copy_path(str(sw_config_data.path),
                             str(self.bundle_data.path.joinpath("config")))
