# -*- coding: utf-8 -*-

import logging
import os

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import process

from sandbox.projects.images.rq import resources
from sandbox.projects.common.mediasearch import ban as mediaban

VAULT = "IMGRQ-YT-TOKEN"
VAULT_DWELLER = "MULTIMEDIA_FUNCTIONALITY"


class MrServerParameter(parameters.SandboxStringParameter):
    name = "mr_server"
    description = "YT proxy"
    default_value = "hahn.yt.yandex.net"
    required = True


class MrUserParameter(parameters.SandboxStringParameter):
    name = "mr_user"
    description = "MapReduce user"
    default_value = "robot-imgrq"
    required = True


class MrTableQueryPairBan(parameters.SandboxStringParameter):
    name = "mr_table_query_pair_ban"
    description = "MR table prefix of ban table"
    default_value = "//home/imgrq/ban/query-pair-ban"
    required = True


class ImagesRqReleaseQueryPairBan(mediaban.ImagesBaseReleaseBanTask):
    type = 'IMAGES_RQ_RELEASE_QUERY_PAIR_BAN'

    input_parameters = mediaban.ImagesBaseReleaseBanTask.input_parameters + (
        MrServerParameter,
        MrUserParameter,
        MrTableQueryPairBan
    )

    trie_resource = resources.IMAGES_RQ_MIDDLESEARCH_QUERY_PAIR_BAN_FILE
    version_resource = resources.IMAGES_RQ_MIDDLESEARCH_QUERY_PAIR_BAN_VERSION

    release_subject = "images/related_queries/middle/querypairban-data-{timestamp}"
    release_comment = "images related query pairs ban file (querypairban)"
    release_mailto = ["related-queries-dev-robot@yandex-team.ru"]
    release_resources = (
        trie_resource, version_resource
    )

    def _build_ban(self):
        version_path = self.abs_path('version')
        with open(version_path, 'w') as version_file:
            self.__process(
                'check-ban',
                args=[
                    '--ban-table', self.ctx[MrTableQueryPairBan.name]
                ],
                stdout=version_file
            )

        if self._skip_build(version_path, self.version_resource):
            logging.info('Nothing to release')
            return 0

        self.create_resource(self.descr, version_path, self.version_resource)

        trie_path = self.abs_path(resources.IMAGES_RQ_MIDDLESEARCH_QUERY_PAIR_BAN_FILE.basename)
        self.__process(
            'build-query-pair-ban',
            args=[
                '--ban-table', self.ctx[MrTableQueryPairBan.name],
                '--trie', trie_path
            ]
        )
        self.create_resource(self.descr, trie_path, self.trie_resource)

        return os.stat(trie_path).st_size

    def __process(self, mode, args=[], stdout=None):
        ban_builder_tool = self._tool(resources.IMAGES_RQ_MIDDLESEARCH_BAN_BUILDER)

        yt_token = self.get_vault_data(VAULT_DWELLER, VAULT)
        process.run_process([
            ban_builder_tool,
            mode,
            "--proxy", self.ctx[MrServerParameter.name],
            "--token", yt_token,
        ] + args, stdout=stdout, outputs_to_one_file=False, log_prefix='ban_builder.{}'.format(mode))


__Task__ = ImagesRqReleaseQueryPairBan
