import logging
import json
import time

import sandbox.common.types.client as ctc
import sandbox.common.types.task as ctt
from sandbox.projects.common import error_handlers as eh

from sandbox.projects import resource_types

#import sandbox.common.types.resource as ctr

from sandbox.projects.saas.backups.DetachServiceIndex import DetachServiceIndex
from sandbox.projects.saas.backups.DetachRtyserverIndex import DetachRtyserverIndex

import sandbox.projects.images.saas_acceptance.resources as saas_acceptance_resources
#import sandbox.projects.images.basesearch.resources as images_resources
#import sandbox.projects.images.models.resources as images_models_resources

from sandbox import sdk2
#from projects.common.search.components import FusionSearch
from sandbox.projects.common.nanny.nanny import NannyClient


class RunFursionRobotAcceptance(sdk2.Task):
    # sdk2.Task overloads
    class Requirements(sdk2.Task.Requirements):
        disk_space = 61440
        client_tags = (
            ctc.Tag.LXC
        )

    class Parameters(sdk2.Task.Parameters):
        nanny_basesearch_service_name = sdk2.parameters.String(
            'Nanny basesearch service name for testing',
            required=True,
            default_value="saas_refresh_acceptance_images_quick_base_sas",
        )
        nanny_distributor_service_name = sdk2.parameters.String(
            'Nanny distributor service name for testing',
            required=True,
            default_value="saas_refresh_acceptance_images_quick_distributors",
        )
        nanny_indexerproxy_service_name = sdk2.parameters.String(
            'Nanny indexerproxy service name testing',
            required=True,
            default_value="saas_refresh_acceptance_images_quick_indexerproxy",
        )
        nanny_prod_basesearch_service_name = sdk2.parameters.String(
            'Nanny basesearch service name to detach index from',
            required=True,
            default_value="saas_refresh_production_images_base_experiment ",
        )
        detached_index = sdk2.parameters.Task(
            'Resource with detached index, if empty, index will be detached from prod basesearch',
            task_type=DetachServiceIndex.type,
            required=False,
        )

    # def __load_tasks_by_shard(self, task_id):
    #    task_obj = channel.sandbox.get_task(task_id)
    #    logging.info("   USHU   task ctx: = \"%s\"", str(dir(task_obj)))
    #    raise Exception('blablabla')
    #    return False

    def __stop_service(self, nanny_client, service_id):
        logging.info('Stopping service \"%s\"...', service_id)

        payload = {
            'type': 'SET_TARGET_STATE',
            'content': {
                'is_enabled': False,
                'comment': 'Disabling to load snapshot from prod instance'
            },
        }
        nanny_client.create_event(service_id, payload)

        logging.info('Stopped service \"%s\"!', service_id)

    def __get_snapshot_state(self, nanny_client, service_id, snapshot_id):
        current_state = nanny_client.get_service_current_state(service_id)
        active_snapshots = current_state['content']['active_snapshots']
        if not active_snapshots:
            ret_state = 'NO_SNAPSHOTS'
        elif active_snapshots[0]['snapshot_id'] != snapshot_id:
            ret_state = 'NOT_FOUND'
        else:
            ret_state = active_snapshots[0]['state']
        logging.debug('State of snapshot %s is %s', snapshot_id, ret_state)
        return ret_state

    def __activate_service(self, nanny_client, service_id):
        runtime_attrs = nanny_client.get_service_runtime_attrs(service_id)
        snapshot_id = runtime_attrs["_id"]

        logging.info('Activating service \"%s\" (snapshot_id = \"%s\")...', service_id, snapshot_id)
        nanny_client.set_snapshot_state(service_id,
                                        snapshot_id=snapshot_id,
                                        state="ACTIVE",
                                        comment='Starting cloned instance',
                                        recipe="common")
        logging.info('Activated service \"%s\" (snapshot_id = \"%s\")!', service_id, snapshot_id)
        return snapshot_id

    def __activate_service_and_wait(self, nanny_client, service_id, time_limit=None):
        snapshot_id = self.__activate_service(nanny_client, service_id)

        start_time = time.time()
        while self.__get_snapshot_state(nanny_client, service_id, snapshot_id) != 'ACTIVE':
            elapsed_time = int(round(time.time() - start_time))
            if time_limit and (elapsed_time > time_limit):
                raise Exception('Release failed due to time limit in {} s'.format(time_limit))

            logging.info('Waiting for new snapshot to become ACTIVE for %s s', elapsed_time)
            time.sleep(10)

    def __stop_all_services(self, nanny_client):
        self.__stop_service(nanny_client, self.Parameters.nanny_basesearch_service_name)
        self.__stop_service(nanny_client, self.Parameters.nanny_distributor_service_name)
        self.__stop_service(nanny_client, self.Parameters.nanny_indexerproxy_service_name)

    def __activate_all_services(self, nanny_client):
        self.__activate_service_and_wait(nanny_client, self.Parameters.nanny_basesearch_service_name)
        self.__activate_service_and_wait(nanny_client, self.Parameters.nanny_distributor_service_name)
        self.__activate_service_and_wait(nanny_client, self.Parameters.nanny_indexerproxy_service_name)

    def __update_shardmap_on_basesearch(self, nanny_client, shardmap_id):
        resources = nanny_client.get_service_resources(self.Parameters.nanny_basesearch_service_name)
        logging.info('resource info \"%s\"', str(resources))

        has_detached_shard_map = False
        for resource in resources['content']['sandbox_files']:
            # if we have resource, then update it
            if resource['resource_type'] == 'SAAS_DETACHED_INDEX_SHARD_MAP':
                resource['resource_id'] = str(shardmap_id)
                resource['task_id'] = str(self.id)
                has_detached_shard_map = True
                break

        # if we don't have resource in nanny files, then add it
        if not has_detached_shard_map:
            resources['content']['sandbox_files'].append(
                {'is_dynamic': False,
                 'local_path': 'detached_prod_index',
                 'task_type': 'RUN_FURSION_ROBOT_ACCEPTANCE',
                 'task_id': str(self.id),
                 'resource_type': 'SAAS_DETACHED_INDEX_SHARD_MAP',
                 'resource_id': str(shardmap_id)}
            )

        resources['comment'] = "Uploading index detached from prod"
        nanny_client.update_service_resources(self.Parameters.nanny_basesearch_service_name, resources)

    # Step2
    def __get_detached_index_task_id(self):
        if self.Parameters.detached_index:
            self.Context.detached_index_task_id = self.Parameters.detached_index.id
        else:
            with self.memoize_stage.create_children:
                task_class = sdk2.Task[DetachServiceIndex.type]
                self.Context.detached_index_task_id = task_class(
                    self,
                    description="Detaching prod index from prod saas to start acceptance saas with ready to merge data",
                    service_name=self.Parameters.nanny_prod_basesearch_service_name,
                    is_backup=True,
                    resource_ttl=7,
                    backups_min_count=18,
                    max_download_speed="300mbps",
                    delay_threshold=8,
                    save_all_indexes=True,  # save only the oldest index
                    correct_replica_pattern='',
                    deploy_refresh_frozen=False,
                    download_timeout=30,
                    GetTorrentTimeout=30,
                    register_iss_shards=False,
                    iss_shardname_template='',
                    yt_table=''
                ).enqueue().id

                raise sdk2.WaitTask(self.Context.detached_index_task_id, ctt.Status.Group.SUCCEED | ctt.Status.Group.BREAK)

    def on_execute(self):
        nanny_token = sdk2.Vault.data('mseifullin', 'nanny-oauth-token')
        nanny_url = "https://nanny.yandex-team.ru/"
        nanny_client = NannyClient(nanny_url, nanny_token)

        self.__stop_all_services(nanny_client)

        if not self.Context.detached_index_task_id:
            self.__get_detached_index_task_id()

        detached_index_task = sdk2.Task.find(
            id=self.Context.detached_index_task_id,
            child=True,
            type=DetachServiceIndex.type
        ).order(-sdk2.Task.id).first()

        rty_detached_index_tasks = sdk2.Task.find(
            parent=detached_index_task,
            child=True,
            type=DetachRtyserverIndex.type
        ).order(-sdk2.Task.id).limit(0)

        listed_sub_tasks = list(rty_detached_index_tasks)

        rty_detached_index_shard_resources = []
        counter = 0
        for rty_detached_index_task in listed_sub_tasks:
            counter = counter + 1
            shard_resource = sdk2.Resource.find(
                task_id=rty_detached_index_task.id,
                type=resource_types.RTYSERVER_SEARCH_DATABASE
            ).order(-sdk2.Task.id).first()
            rty_detached_index_shard_resources.append(shard_resource)

        shard_to_rbtorrent = {}

        for shard_resource in rty_detached_index_shard_resources:
            eh.ensure(shard_resource.shard not in shard_to_rbtorrent, "Duplicate shard id \"{}\" in the detach index task (\"{}\")".format(shard_resource.shard, detached_index_task.id))
            shard_to_rbtorrent[shard_resource.__attrs__["shard_id_num"]] = shard_resource.skynet_id

        resource = saas_acceptance_resources.SaasDetachedIndexShardMap(self, "Handcrafted shard map for saas index", "shardmap.json", ttl=30)
        resource_data = sdk2.ResourceData(resource)
        resource_data.path.write_bytes(json.dumps(shard_to_rbtorrent))
        resource_data.ready()

        self.__update_shardmap_on_basesearch(nanny_client, resource.id)
        self.__activate_all_services(nanny_client)
