# -*- coding: utf-8 -*-

import logging
from sandbox import sdk2

import sandbox.common.types.task as task_types
import sandbox.common.types.resource as task_resources
from sandbox import common
from sandbox.common.types.misc import NotExists
from sandbox.common.types.task import ReleaseStatus
from sandbox.projects.images.tags.ImagesTagsReleaseAliceBans import ImagesTagsReleaseAliceBans
from sandbox.projects.images.tags.ImagesTagsUpdateBansWizardData import ImagesTagsUpdateBansWizardData
from sandbox.projects.images.tags import resources as tags_resources

import sandbox.sandboxsdk.errors as sdk_errors
from sandbox.sandboxsdk.svn import Arcadia

_WAIT_STATUSES = task_types.Status.Group.FINISH + task_types.Status.Group.BREAK


def MakeFooterLine(description, sb_id, sb_type='task', comment=''):
    sb_url = '{}/{}/{}/view'.format(common.utils.server_url(), sb_type, sb_id)
    sb_url = '<a href="{}">{}</a>'.format(sb_url, sb_url)
    return [description, sb_url, comment]


class ImagesTagsBuildAndCommitAliceBans(sdk2.Task):
    """
        Собирает баны тегов для Алисы и релизит ресурс. В случае успеха вызывает обновление ID ресурсов во фреше и шарде.
    """
    class Requirements(sdk2.Task.Requirements):
        disk_space = 1 * 1024  # 1 Gb
        ram = 1 * 1024  # 1 Gb
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        arcadiaTagBansSourcesPath = sdk2.parameters.SvnUrl(
            'Path to tags bans sources TSV on svn',
            default_value=Arcadia.trunk_url('search/wizard/data/fresh/ImgTagsBan/sources.tsv')
            )
        bansCompilerResource = sdk2.parameters.LastReleasedResource(
            'Bans compiler binary resource',
            resource_type=tags_resources.IMAGES_TAGS_BAN_COMPILER_EXECUTABLE,
            state=(task_resources.State.READY),
            required=True,
            )
        arcadiaFreshTagBansDataPath = sdk2.parameters.SvnUrl(
            'Path to ImgTagsBan fresh data',
            default_value=Arcadia.trunk_url('search/wizard/data/fresh/ImgTagsBan'),
            )
        arcadiaRegularTagBansDataPath = sdk2.parameters.SvnUrl(
            'Path to ImgTagsBan common data',
            default_value=Arcadia.trunk_url('search/wizard/data/wizard/ImgTagsBan'),
            )
        commitMessage = sdk2.parameters.String(
            'Bans data commit message',
            required=True,
        )

    def on_execute(self):
        if self.Context.footer_messages is NotExists:
            self.Context.footer_messages = []

        with self.memoize_stage.build_bans_from_trunk(commit_on_entrance=False):
            build_task = ImagesTagsReleaseAliceBans(
                self,
                description='Build tag bans from {}'.format(self.Parameters.arcadiaTagBansSourcesPath),
                sourcesPath=self.Parameters.arcadiaTagBansSourcesPath,
                bansCompilerResource=self.Parameters.bansCompilerResource
                ).enqueue()
            self.Context.build_task_id = build_task.id
            self.Context.footer_messages.append(MakeFooterLine("Resource build task", self.Context.build_task_id))
            raise sdk2.WaitTask([build_task], _WAIT_STATUSES, wait_all=True)

        with self.memoize_stage.release_build_bans(commit_on_entrance=False):
            resource_build_task = sdk2.Task[self.Context.build_task_id]
            logging.info("resource_build_task: {}".format(resource_build_task))
            logging.info("resource_build_task.status: {}".format(resource_build_task.status))
            if resource_build_task.status == task_types.Status.SUCCESS:
                self.server.release(
                    task_id=resource_build_task.id,
                    type=ReleaseStatus.STABLE,
                    subject="Automatic release for built resource (task #{})".format(self.id)
                )
                raise sdk2.WaitTask([resource_build_task], _WAIT_STATUSES)
            elif resource_build_task.status != task_types.Status.RELEASED:
                raise sdk_errors.SandboxTaskFailureError("Resource build subtask failed.")

        resource_build_task = sdk2.Task[self.Context.build_task_id]
        resource_gzt = sdk2.Resource.find(task=resource_build_task, state=task_resources.State.READY, type=tags_resources.IMAGES_TAGS_BAN_GZT).first()
        resource_pire = sdk2.Resource.find(task=resource_build_task, state=task_resources.State.READY, type=tags_resources.IMAGES_TAGS_BAN_PIRE).first()

        with self.memoize_stage.print_resource_ids(commit_on_entrance=False):
            self.Context.footer_messages.append(MakeFooterLine("Resource {}".format(resource_gzt.path), resource_gzt.id, sb_type='resource'))
            self.Context.footer_messages.append(MakeFooterLine("Resource {}".format(resource_pire.path), resource_pire.id, sb_type='resource'))

        # Commit to fresh data
        with self.memoize_stage.commit_ids_fresh(commit_on_entrance=False):
            commit_task = ImagesTagsUpdateBansWizardData(
                self,
                description='Update to bans. Resource was built in task {}'.format(self.Context.build_task_id),
                ArcadiaDataPath=self.Parameters.arcadiaFreshTagBansDataPath,
                ResourceIds={resource_gzt.path: resource_gzt.id, resource_pire.path: resource_pire.id},
                CommitMessage=self.Parameters.commitMessage
                ).enqueue()
            self.Context.fresh_commit_task_id = commit_task.id
            self.Context.footer_messages.append(MakeFooterLine("Resource commit to fresh task", self.Context.fresh_commit_task_id))
            raise sdk2.WaitTask([commit_task], _WAIT_STATUSES, wait_all=True)

        with self.memoize_stage.chech_fresh_data_commit(commit_on_entrance=False):
            if not self.Context.fresh_commit_task_id is NotExists:
                commit_task = sdk2.Task[self.Context.fresh_commit_task_id]
                if commit_task.status != task_types.Status.SUCCESS:
                    raise sdk_errors.SandboxTaskFailureError("Could not commit updated resource IDs, task failed.")
            else:
                raise sdk_errors.SandboxTaskFailureError("Commit task ID for fresh wizard data is not set.")

        # Commit to wizard data
        with self.memoize_stage.commit_ids_data(commit_on_entrance=False):
            commit_task = ImagesTagsUpdateBansWizardData(
                self,
                description='Update to bans. Resource was built in task {}'.format(self.Context.build_task_id),
                ArcadiaDataPath=self.Parameters.arcadiaRegularTagBansDataPath,
                ResourceIds={resource_gzt.path: resource_gzt.id, resource_pire.path: resource_pire.id},
                CommitMessage=self.Parameters.commitMessage
                ).enqueue()
            self.Context.data_commit_task_id = commit_task.id
            self.Context.footer_messages.append(MakeFooterLine("Resource commit to data task", self.Context.data_commit_task_id))
            raise sdk2.WaitTask([commit_task], _WAIT_STATUSES, wait_all=True)

        with self.memoize_stage.chech_data_commit(commit_on_entrance=False):
            if not self.Context.data_commit_task_id is NotExists:
                commit_task = sdk2.Task[self.Context.data_commit_task_id]
                if commit_task.status != task_types.Status.SUCCESS:
                    raise sdk_errors.SandboxTaskFailureError("Could not commit updated resource IDs, task failed.")
            else:
                raise sdk_errors.SandboxTaskFailureError("Commit task ID for fresh wizard data is not set.")

        return

    def on_exception(self):
        subtasks = list(self.find().limit(0))
        for subtask in subtasks:
            try:
                subtask.stop()
            except sdk_errors.TaskError:
                pass

    @sdk2.footer()
    def footer(self):
        if not self.Context.footer_messages is NotExists:
            style = '<style>table.TSKFTR, th.TSKFTR, td.TSKFTR { border: 1px solid black; padding: 5px }</style>'
            table_header = '<tr>' + ''.join(['<th class="TSKFTR">{}</th>'.format(c) for c in ['Description', 'URL', 'Comments']]) + '</tr>'
            table_content = []
            for row in self.Context.footer_messages:
                table_row = '<tr>' + ''.join(['<td class="TSKFTR">{}</td>'.format(c) for c in row]) + '</tr>'
                table_content.append(table_row)
            table = '<table class="TSKFTR">' + table_header + '\n'.join(table_content) + '<table>'

            return style + table
        else:
            return ''
