import logging
import socket
import time
from sandbox import sdk2
from sandbox.projects import resource_types
import sandbox.projects.common.build.parameters as build_params
from sandbox.projects.common.search import settings
import sandbox.common.types.task as ctt
from sandbox.sandboxsdk.errors import SandboxTaskFailureError


class ImagesCheckoutParameter(build_params.CheckoutParameter):
    description = 'Use selective checkout'
    default_value = False  # Temp fix to disable selective checkout until all bugs are fixed


def set_default_selective_checkout_to_true(input_parameters):
    def __amend_input_parameter(input_parameter):
        if input_parameter == build_params.CheckoutParameter:
            return ImagesCheckoutParameter
        return input_parameter

    return [__amend_input_parameter(input_parameter) for input_parameter in input_parameters]


def images_database_search(task, data):
    argtype = data[0]
    value = data[1]
    if argtype == "resource":
        return argtype, value
    elif argtype == "task":
        index_type = value[0]
        task_id = value[1]
        resource = sdk2.Resource.find(task=sdk2.Task[task_id],
                                      type=settings.ImagesSettings.basesearch_database_resource(index_type=index_type),
                                      state='READY').first()
        if resource is None:
            raise SandboxTaskFailureError("Required database not found in resources of task {}".format(task_id))
        return "resource", resource.id
    elif argtype == "shard":
        # Download shard using IMAGES_LOAD_BASESEARCH_DATABASE task
        index_type = value[0]
        shard_name = value[1]
        logging.info("Shard {} not exists in Sandbox, it will be downloaded".format(shard_name))

        # TODO: don't search shard_checker on each call
        shard_checker = sdk2.Resource.find(type=resource_types.IMGSEARCH_SHARD_CHECKER_EXECUTABLE, attrs=dict(released='stable'), state='READY').first()
        if shard_checker is None:
            raise SandboxTaskFailureError("Released stable shard_checker not found")

        task = sdk2.Task["IMAGES_LOAD_BASESEARCH_DATABASE"](task,
                                                            description="Download shard {}".format(shard_name),
                                                            index_type=index_type,
                                                            shard_name=shard_name,
                                                            shard_checker=shard_checker).enqueue()
        return "task", (index_type, task.id)
    else:
        raise SandboxTaskFailureError("Invalid input data: {}".format(data))


# data is [element]
# element is ('task', (index_type:string, task_id:integer)) or ('resource', resource_id:integer)
def images_database_get_resources(data):
    pending_tasks = filter(lambda x: x[0] == "task", data)
    if pending_tasks:
        raise sdk2.WaitTask(map(lambda x: x[1][1], pending_tasks), [ctt.Status.Group.FINISH, ctt.Status.Group.BREAK], wait_all=True)
    else:
        return map(lambda x: sdk2.Resource.find(id=x[1]).first(), data)


def wait_for_port_open(host, port, timeout):
    logging.info("Wait for {}:{} can accept TCP connections".format(host, port))
    s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    i = 0
    while i < timeout:
        try:
            s.connect((host, port))
            break
        except Exception:
            pass
        i = i + 1
        time.sleep(1)

    s.close()
    if i < timeout:
        logging.info("{}:{} is ready!".format(host, port))
    else:
        raise SandboxTaskFailureError("{}:{} still unaccessible".format(host, port))


def wait_for_context_value(client, task_id, name, timeout):
    logging.info("Wait for {} in context of task {}".format(name, task_id))
    i = 0
    ie = timeout / 6
    while i < ie:
        context = client.task[task_id].context.read()
        value = context.get(name, None)
        if value is not None:
            break
        time.sleep(6)
        i = i + 1

    if i < ie:
        logging.info("{}:{} == {}".format(task_id, name, value))
        return value
    else:
        raise SandboxTaskFailureError("{}:{} still unavailable".format(task_id, name))
