# coding: utf-8
import logging
import time

from sandbox import sdk2
from sandbox.common.types import client as ctc
import sandbox.common.types.misc as ctm

from sandbox.projects.impulse.SastWorkflow.checkout import CheckoutOperation
from sandbox.projects.impulse.SastWorkflow.operation import Operation
from sandbox.projects.impulse.ArcadiaSast.impulse import Impulse
from sandbox.projects.impulse.ArcadiaSast import helpers
import sandbox.projects.sandbox.resources as sb_resources


class ImpulseResource(sdk2.Resource):
    analyser = sdk2.parameters.String("analyser")


class ArcadiaSast(sdk2.Task):
    subtask_binary_path = None

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.GENERIC | ctc.Tag.MULTISLOT
        disk_space = 4096
        ram = 2048
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        checkout_arcadia_from_url = sdk2.parameters.String('Arcadia URL', required=True)
        secret = sdk2.parameters.String('CI secret ID with impulse.token', required=True)
        target = sdk2.parameters.String('Target', required=True)
        organization_id = sdk2.parameters.Integer('imPulse organization ID', required=True)
        project_id = sdk2.parameters.Integer('imPulse project ID', required=True)
        analysers = sdk2.parameters.List('Analysers to run (http://wiki/security/imPulse)', required=True)
        codeql_lang = sdk2.parameters.String('CodeQL language', default='cpp', required=False)
        codeql_qls = sdk2.parameters.String('CodeQL qls pack', required=False)
        include_patterns = sdk2.parameters.List('Include patterns', default=[])
        exclude_patterns = sdk2.parameters.List('Exclude patterns', default=[])

        _container = sdk2.parameters.Container('Environment container resource',
                                               resource_type=sb_resources.LXC_CONTAINER,
                                               attrs=dict(target='impulse'),
                                               required=True)

        with sdk2.parameters.Output():
            with sdk2.parameters.Group('Output') as config_block:
                resulting_reports = sdk2.parameters.List('Resulting reports')

    def on_prepare(self):
        self.analysers = list()
        for analyser in self.Parameters.analysers:
            op = Operation(analyser)
            try:
                op.prepare()
            except AssertionError:
                logging.error("Could not download resource for %s", analyser)
                continue
            self.analysers.append(op)

        arcadia_url, revision = helpers.normalize_arcadia_url(self.Parameters.checkout_arcadia_from_url,
                                                              self.Parameters.target)
        if arcadia_url is None:
            raise Exception("Could not normalize Arcadia URL")
        self.repositories = [{
            'url': arcadia_url,
            'revision': revision,
        }]
        task_parameters = {
            'repositories': self.repositories,
            'include_patterns': self.Parameters.include_patterns,
            'exclude_patterns': self.Parameters.exclude_patterns,
        }
        secret = sdk2.yav.Secret(self.Parameters.secret)
        self.impulse_token = secret.data().get('impulse.token')
        if self.impulse_token is None:
            raise Exception("Could not find impulse.token, read http://wiki/security/imPulse/")
        self.impulse = Impulse(base_url='https://impulse.sec.yandex-team.ru/api/v1', token=self.impulse_token)
        self.task_id = self.impulse.get_task_id(self.Parameters.organization_id,
                                                self.Parameters.project_id,
                                                task_parameters,
                                                self.Parameters.analysers,
                                                sdk2.Task.current.id)
        logging.info("Task ID %s", self.task_id)

    def on_execute(self):
        c = CheckoutOperation(self.repositories)
        c.checkout()

        running_analysers = list()
        for analyser in self.analysers:
            analyser.run(self.Parameters, task_id=self.task_id, path=c.wd, meta=c.meta_path)
            running_analysers.append(analyser.name)

        while True:
            if len(running_analysers) == 0:
                break

            for analyser in self.analysers:
                if analyser.is_ready() and analyser.name in running_analysers:
                    analyser.get_result()
                    running_analysers.remove(analyser.name)

        while True:
            scan_instances = self.impulse.get_scan_instances(self.task_id)
            if len(scan_instances) >= len(self.analysers):
                break
            time.sleep(5)

        if scan_instances:
            reports = list()
            for scan_instance in scan_instances:
                report_url = 'https://impulse.sec.yandex-team.ru/organizations/{}/projects/{}/scans/{}/instances/{}'.format(
                    self.Parameters.organization_id,
                    self.Parameters.project_id,
                    scan_instance['scan_type_name'],
                    scan_instance['id'],
                )
                reports.append(report_url)
            self.Parameters.resulting_reports = reports
