# coding: utf-8

import logging
import os
import re
import tempfile
import uuid

from sandbox import sdk2
from sandbox.sdk2 import yav
# from sandbox.sandboxsdk import environments
from sandbox.common.types import client as ctc
import sandbox.projects.sandbox.resources as sb_resources
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.helpers import ProcessLog

from sandbox.projects.impulse.SastWorkflow.checkout import CheckoutOperation
from sandbox.projects.impulse.BuildCodeQLIndex.codeql import CodeQL


def tag_to_language(tag):
    tag_to_language_dict = {
        "Python": "python",
        "Jupyter Notebook": "python",
        "C++": "cpp",
        "C": "cpp",
        "JavaScript": "javascript",
        "TypeScript": "javascript",
        "Vue": "javascript",
        "Go": "go",
        "Java": "java",
        "C#": "csharp",
    }
    return tag_to_language_dict.get(tag)


class BuildCodeqlIndex(sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.GENERIC | ctc.Tag.MULTISLOT
        disk_space = 4096
        ram = 2048
        # environments =
        #     environments.PipEnvironment("boto3"),
        #     environments.PipEnvironment("requests"),
        # ]

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 43200
        repo_url = sdk2.parameters.String('Repository url', required=True)
        languages = sdk2.parameters.String('Languages (comma separated)', required=False)
        build_command = sdk2.parameters.String('Build command', required=False)
        yav_secret = sdk2.parameters.String('Yav secret id', default="sec-01eqk4dqhhz3bt0es4x1bsqszg")
        suspend_on_execute = sdk2.parameters.Bool('Suspend task on execute', default=False)
        organization_id = sdk2.parameters.Integer('Organization ID', required=True)
        project_id = sdk2.parameters.Integer('Project ID', required=True)
        tag = sdk2.parameters.String('Codeql index tag', default="latest")

        _container = sdk2.parameters.Container('Environment container resource',
                                               resource_type=sb_resources.LXC_CONTAINER,
                                               attrs=dict(target='impulse'),
                                               required=True)

    def process_language(self, language, c, build_command):
        import boto3
        import requests

        secrets = yav.Secret(self.Parameters.yav_secret)
        access_key_id = secrets.data()["ACCESS_KEY_ID"]
        secret_access_key = secrets.data()["SECRET_ACCESS_KEY"]
        impulse_oauth_token = secrets.data()["robot-impulse-oauth-impulse"]

        # get revision
        project_path = list(c.folders.keys())[0]
        revision = c.folders[project_path]["revision"]
        is_arcadia = c.folders[project_path]["is_arcadia"]
        logging.debug("[+] project_path: {}. revision: {}.".format(project_path, revision))

        # prepare temp folder
        temp_dir = tempfile.mkdtemp()
        tarball_path = os.path.join(temp_dir, "codeqldb_{}.tar.gz".format(language))

        # create and upgrade database
        codeql = CodeQL(arcadia_root=c.arcadia_mount_point, temp_dir=temp_dir)
        db_path = codeql.create_database(language, project_path, is_arcadia, build_command)
        if not db_path:
            logging.debug("[+] unable to create codeql db. skip net steps")
            return
        # success = codeql.upgrade_database(db_path)
        # if not success:
        #     logging.debug("[+] unable to update codeql db. skip net steps")
        #     return

        if self.Parameters.suspend_on_execute:
            self.suspend()

        # # debug
        # # FIXME: remove
        # with ProcessLog(self, logger='ls_tmp') as pl:
        #     cmd = ["ls", temp_dir]
        #     sp.Popen(cmd, stdout=pl.stdout, stderr=pl.stderr).wait()

        # create tarball
        with ProcessLog(self, logger='tarball_codeqldb') as pl:
            cmd = ["tar", "-czvf", tarball_path, "-C", db_path, "."]
            sp.Popen(cmd, stdout=pl.stdout, stderr=pl.stderr).wait()

        # s3 part

        # Prepare s3 client

        s3client = boto3.client(
            's3',
            endpoint_url='https://s3.mds.yandex.net',
            aws_access_key_id=access_key_id,
            aws_secret_access_key=secret_access_key
            # verify="/etc/ssl/certs/YandexInternalRootCA.pem"
        )

        # PUSH TARBALL TO S3
        f = open(tarball_path, "rb")
        data = f.read()
        f.close()

        bucket_name = "codeqlindex"
        key = "{}_{}.tar.gz".format(str(uuid.uuid4()), language)
        r = s3client.put_object(Bucket=bucket_name, Body=data, Key=key)

        rc = r.get("ResponseMetadata").get("HTTPStatusCode")
        if rc != 200:
            logging.debug("[+] s3 didnt load. ret status: {}. expected 200".format(rc))
            return
        else:
            logging.debug("[+] successfully uploaded to s3.")

        mds_url = "https://codeqlindex.s3.mds.yandex.net/{}".format(key)

        # REMOVE OLD AND UPDATE NEW MDS URL TO TARBALL

        base_url = "https://impulse.sec.yandex-team.ru/api/v1/codeql/organizations/"
        auth_header = {"Authorization": "OAuth {}".format(impulse_oauth_token)}
        org_id = str(self.Parameters.organization_id)
        proj_id = str(self.Parameters.project_id)
        lang = re.sub("[^a-zA-Z0-9]", "", language)
        tag = re.sub("[^a-zA-Z0-9]", "", self.Parameters.tag)
        url = base_url + "{}/projects/{}/languages/{}/tags/{}".format(org_id, proj_id, lang, tag)

        # delete existing database from index
        r = requests.get(url, headers=auth_header)
        if r.status_code == 200 and r.json().get("ok") and r.json().get("result"):
            current_mds_url = r.json().get("result").get("mds_url")
            current_mds_key = current_mds_url.rsplit("/", 1)[-1]
            s3client.delete_object(Bucket=bucket_name, Key=current_mds_key)
            logging.debug("[+] delete from s3 key")

        data = {"mds_url": mds_url, "revision": revision}

        r = requests.put(url, json=data, headers=auth_header)
        logging.debug("[+] impulse put result code: {}".format(r.status_code))

    def on_execute(self):
        import requests

        # checkout repository
        repository = [{"url": self.Parameters.repo_url}]
        c = CheckoutOperation(repository)
        # with sdk2.ssh.Key(self, key_owner='BUILD_CODEQL_INDEX_SSH_KEY'):
        with sdk2.ssh.Key(self, key_owner='IMPULSE_SSH_KEY'):
            c.checkout()
        logging.debug("[+] checkout success: {}.".format(c.folders))

        # DETERMINE PROJECT LANGUAGE
        languages = self.Parameters.languages
        if languages:
            languages_ = [lang.strip() for lang in languages.split(",")]
            languages = [lang for lang in languages_ if lang in ["python", "cpp", "javascript", "go", "java", "csharp"]]
            languages = list(set(languages))
            logging.debug("[+] get language from parameters: {}".format(languages))

        if not languages:
            secrets = yav.Secret(self.Parameters.yav_secret)
            impulse_oauth_token = secrets.data()["robot-impulse-oauth-impulse"]
            auth_header = {"Authorization": "OAuth {}".format(impulse_oauth_token)}
            org_id = self.Parameters.organization_id
            proj_id = self.Parameters.project_id
            url = "https://impulse.sec.yandex-team.ru/api/v1/storage/organizations/{}/projects/{}".format(org_id, proj_id)
            r = requests.get(url, headers=auth_header)
            if r.status_code != 200:
                logging.debug("[+] inpulse. get project_info status_code: {}".format(r.status_code))
            else:
                j = r.json()
                tags = j["result"]["tags"]
                languages = [tag_to_language(tag) for tag in tags]
                languages = [lang for lang in languages if lang]
                languages = list(set(languages))
                logging.debug("[+] get language from project tags: {}".format(languages))
        if not languages:
            languages = ["python", "cpp", "javascript", "go", "java", "csharp"]
            logging.debug("[+] get language from default predefined list: {}".format(languages))

        for language in languages:
            logging.debug("[+] process language: {}".format(language))
            build_command = self.Parameters.build_command
            self.process_language(language, c, build_command)
