# coding: utf-8

import os
import logging

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.helpers import ProcessLog


class CodeQL:
    def __init__(
        self,
        codeql_home="/usr/local/bin/codeql",
        compile_languages=["java", "cpp", "csharp"],
        noncompile_languages=["go", "javascript", "python"],
        temp_dir="/tmp",
        arcadia_root=None
    ):
        self.codeql_home = codeql_home
        self.compile_languages = compile_languages
        self.noncompile_languages = noncompile_languages
        self.temp_dir = temp_dir
        self.arcadia_root = arcadia_root
        self.codeql_cli_path = self.codeql_home + "/codeql-cli/codeql"

    def _build_cmd(self, language, source_path, db_path, command=None):
        cmd = [self.codeql_cli_path, "database", "create", "--language", language, "--source-root", source_path]
        if command:
            cmd += ["--command", command]
        cmd += [db_path]
        return cmd

    def _build_db_path(self, language, prefix="codeqldb_", postfix="_default"):
        db_name = ""
        if prefix:
            db_name += prefix
        if language:
            db_name += language
        if postfix:
            db_name += postfix
        return os.path.join(self.temp_dir, db_name)

    def _build_logger_name(self, language, prefix="codeql_create_", postfix="_default"):
        logger_name = ""
        if prefix:
            logger_name += prefix
        if language:
            logger_name += language
        if postfix:
            logger_name += postfix
        return logger_name

    def _subprocess(self, cmd, logger_name, env=None):
        with ProcessLog(sdk2.Task.current, logger=logger_name) as pl:
            ret_code = sp.Popen(cmd, stdout=pl.stdout, stderr=pl.stderr, env=env).wait()
        return ret_code

    def _get_env_with_arcadia_gopath(self):
        arcadia_gopath = os.path.split(self.arcadia_root)
        if arcadia_gopath[1]:
            arcadia_gopath = arcadia_gopath[0]
        else:
            arcadia_gopath = os.path.split(arcadia_gopath[0])[0]
        arcadia_gopath = os.path.split(arcadia_gopath)[0]

        go_env = os.environ.copy()
        if "GOPATH" in go_env:
            go_env["GOPATH"] = arcadia_gopath + ":" + go_env["GOPATH"]
        else:
            go_env["GOPATH"] = arcadia_gopath

        return go_env

    def create_database(self, language, source_path, is_arcadia, command=None):

        ret_code = 0

        ya_executable_path = os.path.join(self.arcadia_root, "ya") if self.arcadia_root else "ya"
        yabin_project_path = os.path.join(self.arcadia_root, "devtools/ya/bin")
        build_yabin_project_cmd = [ya_executable_path, "make", "-r", yabin_project_path, "-I", self.temp_dir]
        yabin_executable_path = os.path.join(self.temp_dir, "ya-bin")

        build_command = None
        if language in self.compile_languages and is_arcadia:
            # DEVTOOLSSUPPORT-8225: Problems with tracing ya make tool,
            # so we use ya-bin executable directly
            # build ya-bin
            if not os.path.exists(yabin_executable_path):
                logger_name = self._build_logger_name(language, prefix="build_yabin_without_musl_")
                self._subprocess(build_yabin_project_cmd, logger_name)

            # APPSECPIPELINE-166: Use custom command if specified
            if command:
                logging.debug("[+] trying to build codeqldb for {} without custom command.".format(language))
                build_command = command
            # otherwise first build attempt - with ya-bin executable
            else:
                logging.debug("[+] trying to build codeqldb for {} without ya-bin.".format(language))
                build_command = "{} --no-respawn make --rebuild --checkout".format(yabin_executable_path)

        db_path = self._build_db_path(language)
        cmd = self._build_cmd(language, source_path, db_path, command=build_command)
        logger_name = self._build_logger_name(language)
        if language == "go":
            go_env = self._get_env_with_arcadia_gopath()
            ret_code = self._subprocess(cmd, logger_name, env=go_env)
        else:
            ret_code = self._subprocess(cmd, logger_name)

        # first build try was successfull. return db_path
        if ret_code == 0:
            return db_path

        # logging.debug("[+] failed to build codeql database for {}.".format(language))

        # try rebuild golang with GOFLAGS=-mod=mod env
        if language == "go":
            logging.debug("[+] trying to rebuild codeql database for {} with special GOFLAGS env.".format(language))
            go_env = self._get_env_with_arcadia_gopath()
            go_env["GOFLAGS"] = "-mod=mod"
            db_path = self._build_db_path(language, postfix="_with_goflags")
            logger_name = self._build_logger_name(language, postfix="_with_goflags")
            cmd = self._build_cmd(language, source_path, db_path)
            ret_code = self._subprocess(cmd, logger_name, env=go_env)
        if ret_code == 0:
            return db_path

        # rebuild compilable project with default command if custom was specified and build failed
        if language in self.compile_languages and is_arcadia and command:
            logging.debug("[+] trying to rebuild codeql database for {} without ya-bin.".format(language))
            build_command = "{} --no-respawn make --rebuild --checkout".format(yabin_executable_path)
            db_path = self._build_db_path(language, postfix="_with_ya_bin")
            cmd = self._build_cmd(language, source_path, db_path, command=build_command)
            logger_name = self._build_logger_name(language, postfix="_with_ya_bin")
            ret_code = self._subprocess(cmd, logger_name)
        if ret_code == 0:
            return db_path

        # try rebuild compilable languages without ya make tool
        if language in self.compile_languages and is_arcadia:
            logging.debug("[+] trying to rebuild codeql database for {} without default ya tool.".format(language))
            build_command = "{} make --rebuild --checkout".format(ya_executable_path)
            db_path = self._build_db_path(language, postfix="_with_ya_tool")
            cmd = self._build_cmd(language, source_path, db_path, command=build_command)
            logger_name = self._build_logger_name(language, postfix="_with_ya_tool")
            ret_code = self._subprocess(cmd, logger_name)

            # if build with ya tool failed. Try to use codeql built-in heuristics to build the project.
            if ret_code:
                logging.debug("[+] trying to rebuild codeql database for {} without ya make tool.".format(language))
                db_path = self._build_db_path(language, postfix="_no_ya_make")
                logger_name = self._build_logger_name(language, postfix="_no_ya_make")
                cmd = self._build_cmd(language, source_path, db_path)
                ret_code = self._subprocess(cmd, logger_name)

        if ret_code == 0:
            return db_path
        else:
            return None

    def upgrade_database(self, db_path):

        cmd = [self.codeql_cli_path, "database", "upgrade", db_path]

        with ProcessLog(sdk2.Task.current, logger='codeql_upgrade') as pl:
            ret_code = sp.Popen(cmd, stdout=pl.stdout, stderr=pl.stderr).wait()

        return ret_code == 0
