# coding: utf-8

import os
import logging
import tempfile

from sandbox import sdk2
from sandbox.common.types import client as ctc
import sandbox.common.types.misc as ctm
# from sandbox.projects import resource_types
import sandbox.projects.sandbox.resources as sb_resources
from sandbox.sdk2 import yav
# from sandbox.sdk2.helpers import subprocess as sp
# from sandbox.sdk2.helpers import ProcessLog

from sandbox.projects.impulse.BuildCodeQLQueries.codeql import CodeQL
from sandbox.projects.impulse.BuildCodeQLQueries.queries_s3_cacher import QueriesS3Cacher
from sandbox.projects.impulse.BuildCodeQLQueries.utils import create_tarball
# from sandbox.projects.impulse.SastWorkflow.checkout import CheckoutOperation


class CodeqlQueriesResource(sdk2.Resource):
    """ CodeQL compiled queries cache """


class BuildCodeqlQueries(sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.GENERIC | ctc.Tag.MULTISLOT
        disk_space = 4096
        ram = 2048
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 43200

        codeql_cli_url = sdk2.parameters.String(
            'Url to codeql-cli-binaries zipfile',
            default="https://github.com/github/codeql-cli-binaries/releases/download/v2.4.2/codeql-linux64.zip",
            required=True)
        codeql_queries_url = sdk2.parameters.String(
            'Url to codeql queries source code zipfile or repo url',
            default="https://github.com/github/codeql/archive/main.zip",
            required=True)
        codeql_go_queries_url = sdk2.parameters.String(
            'Url to codeql go queries source code zipfile or repo url',
            default="https://github.com/github/codeql-go/archive/main.zip",
            required=True)
        # codeql_yandex_queries_url = sdk2.parameters.String(
        #     'Url to codeql custom yandex queries source code zipfile',
        #     default="https://github.yandex-team.ru/security/codeql-queries/archive/master.zip",
        #     required=True)
        qls_list = sdk2.parameters.String("List of qls files to compile (comma separated)", required=False)
        yav_secret = sdk2.parameters.String('Yav secret id', default="sec-01eqk4dqhhz3bt0es4x1bsqszg")
        suspend_on_execute = sdk2.parameters.Bool('Suspend task on execute', default=False)

        _container = sdk2.parameters.Container('Environment container resource',
                                               resource_type=sb_resources.LXC_CONTAINER,
                                               attrs=dict(target='impulse'),
                                               required=True)

    def on_execute(self):

        secrets = yav.Secret(self.Parameters.yav_secret)
        access_key_id = secrets.data()["ACCESS_KEY_ID"]
        secret_access_key = secrets.data()["SECRET_ACCESS_KEY"]

        codeql_home = tempfile.mkdtemp()
        logging.debug("[+] codeql_home: {}.".format(codeql_home))

        # repositories = [{"url": self.Parameters.codeql_custom_queries_url}]
        # c = CheckoutOperation(repositories)
        # with sdk2.ssh.Key(self, key_owner='BUILD_CODEQL_INDEX_SSH_KEY'):
        #     c.checkout()
        # logging.debug("[+] checkout custom queries folders: {}.".format(c.folders))
        # codeql_yandex_queries_folder = list(c.folders.keys())[0]

        codeql = CodeQL(codeql_home=codeql_home)
        ret_code = codeql.prepare_env(
            codeql_cli_url=self.Parameters.codeql_cli_url,
            codeql_queries_url=self.Parameters.codeql_queries_url,
            codeql_go_queries_url=self.Parameters.codeql_go_queries_url,
            # codeql_yandex_queries_url=self.Parameters.codeql_yandex_queries_url
        )
        if ret_code:
            raise Exception("[+] Error in preparing codeql environment")

        if self.Parameters.suspend_on_execute:
            self.suspend()

        ###
        ###
        ###

        logging.debug("[+] Compiling queries")
        qls_list = list()
        if self.Parameters.qls_list:
            qls = self.Parameters.qls_list.split(",")
            qls_list = [x.strip() for x in qls]
        else:
            for lang in ["cpp", "java", "python", "javascript", "go"]:
                for qls_pack_suffix in ["secaudit.qls", "security-extended.qls"]:
                    qls_list.append("{}-{}".format(lang, qls_pack_suffix))
        codeql.compile_queries(qls_list)

        if self.Parameters.suspend_on_execute:
            self.suspend()

        ###
        ###
        ###

        logging.debug("[+] Creating tarballs queries")

        codeql_queries_tarball_path = os.path.join(codeql_home, "codeql-queries.tar.gz")
        codeql_go_queries_tarball_path = os.path.join(codeql_home, "codeql-go-queries.tar.gz")
        # codeql_yandex_queries_tarball_path = os.path.join(codeql_home, "codeql-yandex-queries.tar.gz")

        logfile = "_".join(["tar", "codeql-queries"])
        path = codeql.codeql_queries_folder_path
        dest = codeql_queries_tarball_path
        create_tarball(path, dest, logfile=logfile)

        logfile = "_".join(["tar", "codeql-go-queries"])
        path = codeql.codeql_go_queries_folder_path
        dest = codeql_go_queries_tarball_path
        create_tarball(path, dest, logfile=logfile)

        # logfile = "_".join(["tar", "codeql-yandex-queries"])
        # path = codeql.codeql_yandex_queries_folder_path
        # dest = codeql_yandex_queries_tarball_path
        # create_tarball(path, dest, logfile=logfile)

        if self.Parameters.suspend_on_execute:
            self.suspend()

        ###
        ###
        ###

        logging.debug("[+] Creating sandbox resources for codeql queries cache")

        codeql_queries_resource = sdk2.ResourceData(CodeqlQueriesResource(
            self, "CodeQL queries cache", "codeql-queries.tar.gz"
        ))
        f = open(codeql_queries_tarball_path, "rb")
        data = f.read()
        f.close()
        codeql_queries_resource.path.write_bytes(data)
        codeql_queries_resource.ready()

        codeql_go_queries_resource = sdk2.ResourceData(CodeqlQueriesResource(
            self, "CodeQL golang queries cache", "codeql-go-queries.tar.gz"
        ))
        f = open(codeql_go_queries_tarball_path, "rb")
        data = f.read()
        f.close()
        codeql_go_queries_resource.path.write_bytes(data)
        codeql_go_queries_resource.ready()

        # codeql_yandex_queries_resource = sdk2.ResourceData(CodeqlQueriesResource(
        #     self, "CodeQL custom queries cache", "codeql-yandex-queries.tar.gz"
        # ))
        # f = open(codeql_yandex_queries_tarball_path, "rb")
        # data = f.read()
        # f.close()
        # codeql_yandex_queries_resource.path.write_bytes(data)
        # codeql_yandex_queries_resource.ready()

        if self.Parameters.suspend_on_execute:
            self.suspend()

        s3cacher = QueriesS3Cacher(access_key_id, secret_access_key)
        queries = [
            (codeql_queries_tarball_path, "codeql-queries.tar.gz"),
            (codeql_go_queries_tarball_path, "codeql-go-queries.tar.gz"),
            # (codeql_yandex_queries_tarball_path, "codeql-yandex-queries.tar.gz"),
        ]
        for tarball_path, name in queries:
            logging.debug("[+] Uploading {} to s3".format(name))
            f = open(tarball_path, "rb")
            data = f.read()
            f.close()
            key = s3cacher.cache_queries(name, data)
            if key:
                logging.debug("[+] Successfully uploaded {} to s3 with key: {}".format(name, key))
            else:
                logging.debug("[+] Error uploading {} to s3.".format(name))
