import logging


class QueriesS3Cacher:

    def __init__(self, access_key_id, secret_access_key, endpoint_url="https://s3.mds.yandex.net", bucket="codeqlindex"):
        import boto3
        self.bucket = bucket
        self.s3client = boto3.client(
            "s3",
            endpoint_url=endpoint_url,
            aws_access_key_id=access_key_id,
            aws_secret_access_key=secret_access_key
        )

    def cache_queries(self, name, data):
        """Uploads <data> to key cache1-<name> or cache2-<name>."""

        cache1_key = "cache1-{}".format(name)
        obj1 = self.get_object(cache1_key)
        if not obj1:
            r = self.put_object(cache1_key, data)
            if r and "ResponseMetadata" in r and r["ResponseMetadata"]["HTTPStatusCode"] == 200:
                return cache1_key
            else:
                logging.debug("[+] error uploading {}.".format(cache1_key))

        cache2_key = "cache2-{}".format(name)
        obj2 = self.get_object(cache2_key)
        if not obj2:
            r = self.put_object(cache2_key, data)
            if r and "ResponseMetadata" in r and r["ResponseMetadata"]["HTTPStatusCode"] == 200:
                return cache2_key
            else:
                logging.debug("[+] error uploading {}.".format(cache2_key))

        older_cache_key = cache1_key if obj1["LastModified"] < obj2["LastModified"] else cache2_key

        r = self.delete_object(older_cache_key)
        if not r or "ResponseMetadata" not in r or r["ResponseMetadata"]["HTTPStatusCode"] != 204:
            logging.debug("[+] error deleting old cache object {}. Retry on another key.".format(older_cache_key))
            raise Exception("Unable to delete old cache object")

        r = self.put_object(older_cache_key, data)
        if r and "ResponseMetadata" in r and r["ResponseMetadata"]["HTTPStatusCode"] == 200:
            return older_cache_key
        else:
            raise Exception("Unable to upload cached queries object")

    def get_object(self, key):
        """
        r["LastModified"]
        r["Body"].read()
        """
        from botocore.exceptions import ClientError
        try:
            r = self.s3client.get_object(Bucket=self.bucket, Key=key)
            return r
        except ClientError as ex:
            logging.debug("[+] s3client ClientError occured: {}".format(ex))
            return None

    def put_object(self, key, data):
        """
        r["ResponseMetadata"]["HTTPStatusCode"] == 200
        """
        from botocore.exceptions import ClientError
        try:
            r = self.s3client.put_object(Bucket=self.bucket, Body=data, Key=key)
            return r
        except ClientError as ex:
            logging.debug("[+] s3client ClientError occured: {}".format(ex))
            return None

    def delete_object(self, key):
        """
        r["ResponseMetadata"]["HTTPStatusCode"] == 204
        """
        from botocore.exceptions import ClientError
        try:
            r = self.s3client.delete_object(Bucket=self.bucket, Key=key)
            return r
        except ClientError as ex:
            logging.debug("[+] s3client ClientError occured: {}".format(ex))
            return None
