# coding: utf-8
import logging

from sandbox import sdk2
from sandbox.common.types import client as ctc
import sandbox.common.types.misc as ctm

from sandbox.projects.impulse.ProjectsCollector.arcadia import Arcadia
from sandbox.projects.impulse.ProjectsCollector.impulse import Impulse
from sandbox.projects.impulse.ProjectsCollector.insights import ProjectInsights
from sandbox.projects.impulse.ProjectsCollector.bitbucket import BitbucketAPI
from sandbox.projects.impulse.ProjectsCollector.github import GitHubAPI
from sandbox.projects.impulse.ProjectsCollector.common import langs2analyzers, langs2scanparams


class ProjectsCollector(sdk2.Task):
    subtask_binary_path = None

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.GENERIC | ctc.Tag.MULTISLOT
        disk_space = 4096
        ram = 2048
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        secret = sdk2.parameters.String('Secrets', required=True)
        enable_github = sdk2.parameters.Bool("Collect internal GitHub projects")
        with enable_github.value[True]:
            github_organization_id = sdk2.parameters.Integer('imPulse Internal GitHub organization ID', default=189)
        enable_bitbucket = sdk2.parameters.Bool("Collect Bitbucket projects")
        with enable_bitbucket.value[True]:
            bitbucket_organization_id = sdk2.parameters.Integer('imPulse Bitbucket organization ID', default=223)
        enable_arcadia = sdk2.parameters.Bool("Collect Arcadia projects")
        with enable_arcadia.value[True]:
            arcadia_organization_id = sdk2.parameters.Integer('imPulse Arcadia organization ID', default=190)
        enable_project_insights = sdk2.parameters.Bool("Collect Arcadia projects insights")
        with enable_project_insights.value[True]:
            project_insights_organization_id = sdk2.parameters.Integer('ImPulse Project Insights organization ID',
                                                                       default=329)
            impulse_tvm_id = sdk2.parameters.Integer('ImPulse Webhook TVM ID', default=2017557)

    def on_prepare(self):
        secret = sdk2.yav.Secret(self.Parameters.secret)

        self.github_token = secret.data().get('github_token')
        if self.Parameters.enable_github and self.github_token is None:
            raise Exception("Could not find github_token")

        self.bitbucket_token = secret.data().get('bitbucket_token')
        if self.Parameters.enable_bitbucket and self.bitbucket_token is None:
            raise Exception("Could not find bitbucket_token")

        self.impulse_token = secret.data().get('impulse_token')
        if self.impulse_token is None:
            raise Exception("Could not find impulse_token")

        self.arcanum_token = secret.data().get('arcanum_token')
        if self.arcanum_token is None:
            raise Exception("Could not find arcanum_token")

    def _github_collector(self, impulse):
        impulse_projects = impulse.get_projects(self.Parameters.github_organization_id)
        if impulse_projects is None:
            raise Exception("Could not get imPulse projects for Internal GitHub")
        slugs = dict()
        for p in impulse_projects:
            slugs[p["slug"]] = p

        gh = GitHubAPI(upstream="https://github.yandex-team.ru/api/v3", token=self.github_token)
        organizations = gh.fetch_organizations()
        logging.info("[github] fetch %d organizations", len(organizations))
        for organization in organizations:
            repos = gh.fetch_repos(organization)
            logging.info("[github][%s] fetch %d repos", organization, len(repos))
            for repo in repos:
                if repo["slug"] in slugs:
                    tags = gh.get_languages(organization, repo["name"])
                    stored_tags = slugs[repo["slug"]].get("tags", [])
                    if stored_tags is None or set(tags) != set(stored_tags):
                        res = impulse.update_project(organization_id=self.Parameters.github_organization_id,
                                                     project_id=slugs[repo["slug"]]["id"],
                                                     name=repo["full_name"],
                                                     tags=tags,
                                                     tracker_queue=slugs[repo["slug"]]["tracker_queue"],
                                                     abc_service_id=slugs[repo["slug"]]["abc_service_id"],
                                                     )
                        if res is None:
                            logging.info("[github] could not update project %s, skipping", repo["slug"])
                    continue
                res = impulse.create_project(organization_id=self.Parameters.github_organization_id,
                                             name=repo["full_name"], slug=repo["slug"],
                                             tags=gh.get_languages(organization, repo["name"]))
                if res is None:
                    logging.info("[github] could not create project %s, skipping", repo["slug"])
                    continue

                scan_params = {
                    "repositories": [
                        {
                            "url": repo["url"],
                            "branch": "",
                        }
                    ]
                }
                scan = impulse.create_scan(organization_id=self.Parameters.github_organization_id,
                                           project_id=res["id"], analysers=[], parameters=scan_params)
                if scan is None:
                    logging.info("[github] could not create scan for %s, skipping", repo["slug"])

    def _bitbucket_collector(self, impulse):
        impulse_projects = impulse.get_projects(self.Parameters.bitbucket_organization_id)
        if impulse_projects is None:
            raise Exception("Could not get imPulse projects for Internal Bitbucket")
        slugs = dict()
        for p in impulse_projects:
            slugs[p["slug"]] = p

        bb = BitbucketAPI(upstream="https://bb.yandex-team.ru/rest/api/1.0", token=self.bitbucket_token)
        projects = bb.fetch_projects()
        logging.info("[bitbucket] fetch %d projects", len(projects))
        for project in projects:
            repos = bb.fetch_repos(project)
            logging.info("[bitbucket][%s] fetch %d repos", project, len(repos))
            for repo in repos:
                if repo["slug"] in slugs:
                    tags = bb.get_languages(project, repo["name"])
                    stored_tags = slugs[repo["slug"]].get("tags", [])
                    if stored_tags is None or set(tags) != set(stored_tags):
                        res = impulse.update_project(organization_id=self.Parameters.github_organization_id,
                                                     project_id=slugs[repo["slug"]]["id"],
                                                     name=repo["full_name"],
                                                     tags=tags,
                                                     tracker_queue=slugs[repo["slug"]]["tracker_queue"],
                                                     abc_service_id=slugs[repo["slug"]]["abc_service_id"],
                                                     )
                        if res is None:
                            logging.info("[bitbucket] could not update project %s, skipping", repo["slug"])
                    continue
                res = impulse.create_project(organization_id=self.Parameters.bitbucket_organization_id,
                                             name=repo["full_name"], slug=repo["slug"],
                                             tags=bb.get_languages(project, repo["name"]))
                if res is None:
                    logging.info("[bitbucket] could not create project %s, skipping", repo["slug"])
                    continue

                scan_params = {
                    "repositories": [
                        {
                            "url": repo["url"],
                            "branch": "",
                        }
                    ]
                }
                scan = impulse.create_scan(organization_id=self.Parameters.bitbucket_organization_id,
                                           project_id=res["id"], analysers=[], parameters=scan_params)
                if scan is None:
                    logging.info("[bitbucket] could not create scan for %s, skipping", repo["slug"])

    def _arcadia_collector(self, impulse):
        impulse_projects = impulse.get_projects(self.Parameters.arcadia_organization_id)
        if impulse_projects is None:
            raise Exception("Could not get imPulse projects for Arcadia")
        slugs = dict()
        for p in impulse_projects:
            slugs[p["slug"]] = p

        a = Arcadia()
        projects = a.fetch_projects()
        logging.info("[arcadia] fetch %d projects", len(projects))
        for project in projects:
            if project["slug"] in slugs:
                stored_tags = slugs[project["slug"]].get("tags", [])
                if stored_tags is None or set(project["languages"]) != set(stored_tags):
                    res = impulse.update_project(organization_id=self.Parameters.arcadia_organization_id,
                                                 project_id=slugs[project["slug"]]["id"],
                                                 name=project["full_name"],
                                                 tags=project["languages"],
                                                 tracker_queue=slugs[project["slug"]]["tracker_queue"],
                                                 abc_service_id=slugs[project["slug"]]["abc_service_id"],
                                                 )
                    if res is None:
                        logging.info("[arcadia] could not update project %s, skipping", project["slug"])
                continue

            # skip = False
            # for slug in slugs.keys():
            #     if project["slug"].startswith(slug):
            #         # check if larger project already exists
            #         skip = True
            #         break
            # if skip:
            #     continue
            # just comment

            # delete larger project if exists
            for slug in slugs.keys():
                if project["slug"].startswith(slug):
                    ok = impulse.delete_project(self.Parameters.arcadia_organization_id, slugs[slug]["id"])
                    if ok:
                        logging.info("[arcadia] deleted project %s", slug)
                    else:
                        logging.info("[arcadia] could not delete project %s", slug)

            res = impulse.create_project(organization_id=self.Parameters.arcadia_organization_id,
                                         name=project["full_name"], slug=project["slug"],
                                         tags=project["languages"])
            if res is None:
                logging.info("[arcadia] could not create project %s, skipping", project["slug"])
                continue

            scan_params = {
                "repositories": [
                    {
                        "url": project["url"],
                        "branch": "",
                    }
                ]
            }
            scan = impulse.create_scan(organization_id=self.Parameters.arcadia_organization_id,
                                       project_id=res["id"], analysers=[], parameters=scan_params)
            if scan is None:
                logging.info("[arcadia] could not create scan for %s, skipping", project["slug"])

    def _project_insights_collector(self, impulse):
        callback_url = "https://impulse.sec.yandex-team.ru/api/v1/webhook/insights/export?tvm_id=%d" % self.Parameters.impulse_tvm_id
        impulse_projects = impulse.get_projects(self.Parameters.project_insights_organization_id)
        if impulse_projects is None:
            raise Exception("Could not get imPulse projects for Project Insights")

        impulse_projects_slugs = set()
        impulse_projects_dict = dict()
        for p in impulse_projects:
            impulse_projects_slugs.add(p["slug"])
            impulse_projects_dict[p["slug"]] = p

        p = ProjectInsights(upstream="https://a.yandex-team.ru", token=self.arcanum_token)
        projects = p.fetch_projects()
        if not projects:
            raise Exception("Could not get Arcanum projects")

        new_projects_slugs = set(p["slug"] for p in projects)

        to_create = new_projects_slugs.difference(impulse_projects_slugs)
        to_update = new_projects_slugs.intersection(impulse_projects_slugs)
        to_delete = impulse_projects_slugs.difference(new_projects_slugs)

        for project in projects:
            if project["slug"] in to_update:
                stored_tags = impulse_projects_dict[project["slug"]].get("tags", [])
                if stored_tags is None or set(project.get("languages", [])) != set(stored_tags):
                    res = impulse.update_project(organization_id=self.Parameters.project_insights_organization_id,
                                                 project_id=impulse_projects_dict[project["slug"]]["id"],
                                                 name=project["full_name"],
                                                 tags=project["languages"],
                                                 tracker_queue=impulse_projects_dict[project["slug"]]["tracker_queue"],
                                                 abc_service_id=impulse_projects_dict[project["slug"]]["abc_service_id"],
                                                 )
                    if res is None:
                        logging.info("[project insights] could not update project %s, skipping", project["slug"])
            elif project["slug"] in to_create:
                res = impulse.create_project(organization_id=self.Parameters.project_insights_organization_id,
                                             name=project["full_name"], slug=project["slug"],
                                             tags=project["languages"])
                if res is None:
                    logging.info("[project insights] could not create project %s, skipping", project["slug"])
                else:
                    impulse_projects.append(res)

            if project["slug"] not in to_delete:
                project_id = impulse_projects_dict.get(project["slug"], {}).get("id")
                analysers = langs2analyzers(project["languages"])
                if project_id and analysers:
                    scan_params = langs2scanparams(project["languages"])
                    scan_params["repositories"] = [{
                        "url": project["url"],
                        "branch": "",
                    }]
                    scan = impulse.create_scan(organization_id=self.Parameters.project_insights_organization_id,
                                               project_id=project_id, analysers=analysers,
                                               parameters=scan_params, callback_url=callback_url)
                    if scan is None:
                        logging.info("[project insights] could not create scan for %s, skipping", project["slug"])
        for project in impulse_projects:
            if project["slug"] in to_delete:
                res = impulse.delete_project(self.Parameters.project_insights_organization_id, project["id"])
                if res is None:
                    logging.info("[project insights] could not delete project with ID %s", project["id"])

    def on_execute(self):
        impulse = Impulse(base_url='https://impulse.sec.yandex-team.ru/api/v1', token=self.impulse_token)

        if self.Parameters.enable_github:
            self._github_collector(impulse)

        if self.Parameters.enable_bitbucket:
            self._bitbucket_collector(impulse)

        if self.Parameters.enable_arcadia:
            self._arcadia_collector(impulse)

        if self.Parameters.enable_project_insights:
            self._project_insights_collector(impulse)
