from requests import get
from time import time

from sandbox.projects.impulse.ProjectsCollector.common import ext2lang


class BitbucketAPI:
    def __init__(self, upstream, token):
        self.headers = {
            "Authorization": "Bearer {}".format(token),
            "Content-Type": "application/json"
        }
        self.upstream = upstream

    def fetch_projects(self):
        upstream = self.upstream + "/projects"
        params = {
            "start": 0,
            "limit": 100,
        }
        projects = []
        while True:
            res = get(upstream, headers=self.headers, params=params)
            if res.status_code != 200:
                break
            data = res.json()
            keys = [d["key"] for d in data.get("values", [])]
            if len(keys) == 0:
                break
            projects += keys
            if data["isLastPage"]:
                break
            params["start"] += data["size"]

        return projects

    def get_modification_timestamp(self, project, repo):
        upstream = self.upstream + "/projects/{}/repos/{}/commits".format(project, repo)
        params = {
            "limit": 1,
        }
        res = get(upstream, headers=self.headers, params=params)
        if res.status_code != 200:
            return 0
        data = res.json()
        values = data.get("values", [])
        if len(values) == 0:
            return 0
        timestamp = values[0]["committerTimestamp"] / 1000

        return timestamp

    def fetch_repos(self, project):
        upstream = self.upstream + "/projects/{}/repos".format(project)
        params = {
            "start": 0,
            "limit": 100,
        }
        repos = []
        while True:
            res = get(upstream, headers=self.headers, params=params)
            if res.status_code != 200:
                break
            data = res.json()
            values = data.get("values", [])
            if len(values) == 0:
                break
            for d in values:
                timestamp = self.get_modification_timestamp(project, d["name"])
                if time() - timestamp > 3600 * 24 * 30:
                    continue
                slug = "bb-{}-{}".format(d["project"]["name"], d["name"])
                name = d["name"]
                full_name = "{}/{}".format(d["project"]["name"], d["name"])
                if len(d['links']['self']):
                    url = d['links']['self'][0]['href']
                else:
                    url = "https://bb.yandex-team.ru/projects/{}/repos/{}/browse".format(d["project"]["name"],
                                                                                         d["name"])
                repos.append(dict(name=name, full_name=full_name, url=url, slug=slug))
            params["start"] += data["size"]

        return repos

    def get_languages(self, project, repo):
        upstream = self.upstream + "/projects/{}/repos/{}/files".format(project, repo)
        params = {
            "start": 0,
            "limit": 1000,
        }
        res = get(upstream, headers=self.headers, params=params)
        if res.status_code != 200:
            return []

        data = res.json()
        exts = list(set(f.rsplit('/', 1)[-1].rsplit('.', 1)[-1] for f in data['values']))

        languages = []
        for ext in exts:
            lang = ext2lang(ext)
            if lang:
                languages.append(lang)

        return languages
