from requests import get
from datetime import datetime


class GitHubAPI:
    def __init__(self, upstream, token):
        self.headers = {
            "Authorization": "token {}".format(token),
            "Content-Type": "application/json"
        }
        self.upstream = upstream

    def fetch_organizations(self):
        upstream = self.upstream + "/organizations"
        params = {
            "per_page": 100,
            "since": 0
        }
        organizations = []
        while True:
            res = get(upstream, headers=self.headers, params=params)
            if res.status_code != 200:
                break
            data = res.json()
            logins = [d["login"] for d in data]
            if len(logins) == 0:
                break
            organizations += logins
            params["since"] = data[-1]["id"] + 1

        return organizations

    def fetch_repos(self, organization):
        upstream = self.upstream + "/orgs/{}/repos".format(organization)
        params = {
            "per_page": 100,
            "page": 1
        }
        repos = []
        while True:
            res = get(upstream, headers=self.headers, params=params)
            if res.status_code != 200:
                break
            data = res.json()
            if len(data) == 0:
                break
            for d in data:
                if d["disabled"] or d["archived"] or d["fork"]:
                    continue
                delta = datetime.now() - datetime.strptime(d["updated_at"], "%Y-%m-%dT%H:%M:%SZ")
                if delta.days > 30:
                    continue
                slug = d["full_name"].replace("/", "-")
                name = d["name"]
                full_name = d["full_name"]
                url = d["html_url"]
                repos.append(dict(name=name, full_name=full_name, url=url, slug=slug))
            params["page"] += 1

        return repos

    def get_languages(self, organization, repo):
        upstream = self.upstream + "/repos/{}/{}/languages".format(organization, repo)
        res = get(upstream, headers=self.headers)
        if res.status_code != 200:
            return []

        data = res.json()
        return list(data.keys())
