# coding: utf-8
import logging

from sandbox import sdk2
from sandbox.common.types import client as ctc

from sandbox.projects.impulse.SastWorkflow.checkout import CheckoutOperation
from sandbox.projects.impulse.SastWorkflow.operation import Operation
import sandbox.projects.sandbox.resources as sb_resources


class SastWorkflow(sdk2.Task):
    subtask_binary_path = None

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.GENERIC | ctc.Tag.MULTISLOT
        disk_space = 4096
        ram = 2048

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 259200  # 3 days
        task_id = sdk2.parameters.String('imPulse task ID', required=True)
        organization_id = sdk2.parameters.Integer('Organization ID', required=True)
        project_id = sdk2.parameters.Integer('Project ID', required=True)
        analysers = sdk2.parameters.List('List of analyzers to run (security/impulse/workflow/static)')
        parameters = sdk2.parameters.JSON('Parameters', required=True)
        verbose = sdk2.parameters.Bool('Verbose', default=False, required=False)

        _container = sdk2.parameters.Container('Environment container resource',
                                               resource_type=sb_resources.LXC_CONTAINER,
                                               attrs=dict(target='impulse'),
                                               required=True)

    def on_prepare(self):
        self.analysers = list()
        for analyser in self.Parameters.analysers:
            op = Operation(analyser, verbose=self.Parameters.verbose)
            try:
                op.prepare()
            except AssertionError:
                logging.error("Could not download resource for %s", analyser)
                continue
            self.analysers.append(op)

    def on_execute(self):
        c = CheckoutOperation(self.Parameters.parameters.get('repositories'))
        running_analysers = list()
        with sdk2.ssh.Key(self, key_owner='IMPULSE_SSH_KEY'):
            c.checkout()
            for analyser in self.analysers:
                analyser.run(self.Parameters, path=c.wd, meta=c.meta_path)
                running_analysers.append(analyser.name)

            while True:
                if len(running_analysers) == 0:
                    break
                for analyser in self.analysers:
                    if analyser.is_ready() and analyser.name in running_analysers:
                        running_analysers.remove(analyser.name)
