# coding: utf-8
import os
import os.path
import tempfile
import logging
from json import dumps
from urlparse import urlparse

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.helpers import ProcessLog

from sandbox.projects.common.vcs import arc


def get_arc_revision(path):
    cmd = ["arc", "rev-parse", "HEAD"]
    output = sp.check_output(cmd, cwd=path, stderr=sp.PIPE)
    return output.strip()


def get_git_revision(path):
    cmd = ["git", "rev-parse", "HEAD"]
    output = sp.check_output(cmd, cwd=path, stderr=sp.PIPE)
    return output.strip()


def set_arc_revision(path, revision_hash):
    cmd = ["arc", "checkout", revision_hash]
    sp.check_output(cmd, cwd=path, stderr=sp.PIPE)


class CheckoutOperation(object):
    arcadia_base_path = 'src/a.yandex-team.ru'

    def __init__(self, repositories):
        self.repositories = repositories
        self.folders = dict()
        self.meta_path = None
        self.wd = os.getcwd()
        self.arcadia_latest_commit = ""

        self.arcadia_mount_point = os.path.join(self.wd, self.arcadia_base_path)
        self.mp = None
        os.makedirs(self.arcadia_mount_point)

    def checkout(self):
        arcadia_branch_set = set()
        for repo in self.repositories:
            repo_url = repo.get('url', '')
            repo_path = repo.get('path', '').lstrip('/')
            repo_branch = repo.get('branch')
            revision = repo.get('revision')

            parsed_url = urlparse(repo_url)
            vcs_domain = parsed_url.netloc.split('@', 1)[-1]

            is_arcadia = False

            if vcs_domain in ["a.yandex-team.ru", "arcadia.yandex.ru"]:
                is_arcadia = True
                if not repo_branch:
                    repo_branch = "trunk"
                arcadia_branch_set.add(repo_branch)
                if len(arcadia_branch_set) > 1:
                    raise ValueError('Different branches in single checkout operation is not supported for Arcadia', arcadia_branch_set)

                rel_path = os.path.join(self.arcadia_base_path, parsed_url.path.split('/', 4)[-1])

                if self.mp is None:
                    self.mp = arc.Arc().mount_path(None, changeset=repo_branch, mount_point=self.arcadia_mount_point, fetch_all=False)

                if revision:
                    set_arc_revision(self.arcadia_mount_point, revision)
                else:
                    revision = get_arc_revision(self.arcadia_mount_point)
                    # Set arcadia latest commit only for trunk because we need Project Insights only for trunk
                    self.arcadia_latest_commit = revision
            elif vcs_domain in ["github.yandex-team.ru", "bb.yandex-team.ru", "bitbucket.sdc.yandex-team.ru", "git.adfox.ru", "github.com",
                                "git.edadeal.yandex-team.ru", "bitbucket.browser.yandex-team.ru"]:
                path_parts = parsed_url.path.split('/')
                org_path = os.path.join('src', vcs_domain, path_parts[-2])
                rel_path = os.path.join('src', vcs_domain, path_parts[-2], path_parts[-1].replace('.git', ''))
                checkout_path = os.path.join(self.wd, rel_path)
                if not os.path.exists(org_path):
                    os.makedirs(org_path)
                if repo_branch:
                    cmd = ["git", "clone", "--depth", "1", "--branch", repo_branch, repo_url, checkout_path]
                else:
                    cmd = ["git", "clone", "--depth", "1", repo_url, checkout_path]
                with ProcessLog(sdk2.Task.current, logger='checkout') as pl:
                    sp.Popen(cmd, stdout=pl.stdout, stderr=pl.stderr).wait()
                revision = get_git_revision(checkout_path)
            else:
                logging.warning("vcs %s is not supported, skipping", vcs_domain)
                continue

            if repo_path:
                rel_path = os.path.join(rel_path, repo_path)

            self.folders[rel_path] = {
                'repository': repo_url,
                'revision': revision,
                'path': repo_path,
                'is_arcadia': is_arcadia,
            }

        self.meta_path = self.generate_meta()

    def generate_meta(self):
        meta = {
            'folders': self.folders,
        }
        logging.info(meta)
        handle, path = tempfile.mkstemp()
        with os.fdopen(handle, 'w') as f:
            f.write(dumps(meta))
        return path
