# coding: utf-8
import os
import os.path
import logging
from json import dumps, loads
from random import randint, seed

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.helpers import ProcessLog

from sandbox.projects.impulse.SastWorkflow import helpers
from sandbox.projects.security.resources import SecurityToolBinary


class Operation(object):
    GLOBAL_TYPE = 'global'
    INPUT_TYPE = 'input'
    OUTPUT_TYPE = 'output'
    SECRET_TYPE = 'secret'
    CONST_TYPE = 'const'

    target_base_path = 'security/impulse/workflow/static'
    resource_type = SecurityToolBinary

    def __init__(self, operation, verbose=False):
        if operation.endswith('_scan'):
            self.name = operation[:-5]
        else:
            self.name = operation
        self.verbose = verbose
        self.result = dict()
        self.manifest = dict()
        self.target_path = os.path.join(self.target_base_path, self.name)
        self.process = None
        self.binary_path = None

        seed()

        manifest_file = "manifest/{}.json".format(self.name)
        manifest_path = os.path.join(os.path.abspath(os.path.dirname(__file__)),
                                     manifest_file)
        if os.path.exists(manifest_path):
            with open(manifest_path, 'r') as f:
                self.manifest = loads(f.read())
        else:
            logging.error("Could not find manifest file %s", manifest_file)

    def prepare(self):
        resource = self.get_resource()
        self.binary_path = helpers.download_resource(resource)
        helpers.chmod_x(self.binary_path)

    def get_additional_env(self):
        return self.manifest.get('env', {})

    def get_env(self):
        env = os.environ.copy()
        additional_env = self.get_additional_env()
        assert isinstance(additional_env, dict), "Additional environment must be instance of dict"
        env.update(additional_env)
        return env

    def get_resource(self):
        return helpers.get_last_released_resource(self.resource_type,
                                                  dict(target=self.target_path))

    def get_result(self):
        logging.info(self.process.returncode)
        return {}

    def get_cmd(self, global_params, **kwargs):
        cmd = [self.binary_path]
        for key, arg in self.manifest.get('args', {}).items():
            if arg['type'] == Operation.GLOBAL_TYPE:
                value = getattr(global_params, arg['value'], None)
                if value is None and hasattr(global_params, 'parameters'):
                    value = global_params.parameters.get(arg['value'])
                if value is None:
                    value = kwargs.get(arg['value'])
                if value is None:
                    continue
                if type(value) in [list, dict]:
                    value = dumps(value)
                else:
                    value = str(value)
                cmd.append('-' + key)
                cmd.append(value)
            elif arg['type'] == Operation.INPUT_TYPE:
                value = kwargs.get(arg['value'])
                if value is None:
                    continue
                cmd.append('-' + key)
                cmd.append(value)
            elif arg['type'] == Operation.SECRET_TYPE:
                try:
                    value = sdk2.Vault.data(arg['value'])
                except:
                    continue
                cmd.append('-' + key)
                cmd.append(value)
            elif arg['type'] == Operation.OUTPUT_TYPE:
                output_path = '/tmp/{}_{}_{}'.format(self.name, arg['value'],
                                                     str(randint(0, 1000)))
                self.result[key] = output_path
                cmd.append('-' + key)
                cmd.append(output_path)
            elif arg['type'] == Operation.CONST_TYPE:
                cmd.append('-' + key)
                cmd.append(arg['value'])
            else:
                raise ValueError('Unknown arg type', arg['type'])

        logging.info(cmd)
        return cmd

    def run(self, global_params, **kwargs):
        if self.verbose:
            with ProcessLog(sdk2.Task.current, logger=self.name) as pl:
                self.process = sp.Popen(self.get_cmd(global_params, **kwargs),
                                        stdout=pl.stdout,
                                        stderr=pl.stderr,
                                        env=self.get_env())
        else:
            self.process = sp.Popen(self.get_cmd(global_params, **kwargs),
                                    env=self.get_env())

    def is_ready(self):
        if self.process is None:
            return True
        try:
            self.process.wait(0.1)
            return True
        except sp.TimeoutExpired:
            return False
