import logging
import os
import shutil
import subprocess
import tempfile

from sandbox import sdk2
from sandbox.sandboxsdk import errors
from sandbox.projects.common import constants as consts
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.infra.common import get_arcadia
from sandbox.projects.infra.resources import PodAgentBinary
from sandbox.projects.infra.resources import PodAgentBinaryWithDebugInfo


class BuildPodAgentBinary(sdk2.Task):
    """
        Build pod_agent binary
    """

    class Parameters(sdk2.Task.Parameters):
        checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl("Svn url for arcadia (if no value in the context is given)", required=True)
        ignore_pod_agent_binary_size_limit = sdk2.parameters.Bool("Ignore pod_agent binary size limit", default_value=False)

        with sdk2.parameters.Output:
            pod_agent_binary_stripped = sdk2.parameters.Resource("Pod agent binary (stripped)", resource_type=PodAgentBinary)
            pod_agent_binary_with_debug_info = sdk2.parameters.Resource("Pod agent binary (with debug info)", resource_type=PodAgentBinaryWithDebugInfo)

    class Context(sdk2.Task.Context):
        data_name_stripped = "pod_agent"
        data_name_with_debug_info = "pod_agent_with_debug_info"

    def on_execute(self):
        self.Parameters.pod_agent_binary_stripped = PodAgentBinary(
            self, self.Parameters.description, self.Context.data_name_stripped, ttl=720
        )
        self.Parameters.pod_agent_binary_with_debug_info = PodAgentBinaryWithDebugInfo(
            self, self.Parameters.description, self.Context.data_name_with_debug_info, ttl=720
        )

        self.build()

    def on_release(self, parameters_):
        logging.debug("Release parameters: %r", parameters_)
        self._send_release_info_to_email(parameters_)

        release_status = parameters_["release_status"]
        release_tag = parameters_.get("release_tag")

        resources = self.mark_released_resources(release_status)
        if release_tag:
            for resource in resources:
                setattr(resource, "released_{}".format(release_tag), True)

    def build(self):
        target = PodAgentBinary.arcadia_build_path
        build_dir = tempfile.mkdtemp()

        arcadia_url = self.Parameters.checkout_arcadia_from_url
        if consts.ARCADIA_URL_KEY in self.Context:
            arcadia_url = getattr(self.Context, consts.ARCADIA_URL_KEY)
        logging.info("current arcadia url is '{}'".format(arcadia_url))

        with get_arcadia(arcadia_url) as arcadia:
            logging.info("build pod_agent binary...")
            sdk.do_build(
                consts.YMAKE_BUILD_SYSTEM,
                source_root=arcadia,
                build_type=consts.RELEASE_BUILD_TYPE,
                targets=[target],
                results_dir=build_dir,
                clear_build=False,
                test=False
            )
            logging.info("pod_agent built")

        target_path = os.path.join(build_dir, target)
        target_stripped_path = os.path.join(build_dir, target + "_stripped")

        logging.info("strip pod_agent binary...")
        subprocess.check_call(
            [
                "strip",
                target_path,
                "-s",
                "-o", target_stripped_path
            ]
        )
        logging.info("pod_agent binary stripped")

        stripped_size = os.path.getsize(target_stripped_path)
        logging.info("pod_agent stripped binary size is '{}'".format(stripped_size))
        if stripped_size > PodAgentBinary.size_limit and not self.Parameters.ignore_pod_agent_binary_size_limit:
            raise errors.SandboxTaskFailureError("pod_agent stripped binary size is too big '{}', limit is '{}'".format(stripped_size, PodAgentBinary.size_limit))

        shutil.move(target_stripped_path, str(self.Parameters.pod_agent_binary_stripped.path))
        shutil.move(target_path, str(self.Parameters.pod_agent_binary_with_debug_info.path))
