import logging
import os
import subprocess
import tempfile

from sandbox import sdk2
from sandbox.projects.common import constants as consts
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.infra.common import get_arcadia
from sandbox.projects.infra.resources import PodAgentTestServiceLayer


class CreatePodAgentTestServiceLayer(sdk2.Task):
    """
        Build pod_agent_test_service and create layer
    """

    class Parameters(sdk2.Task.Parameters):
        arcadia_url = sdk2.parameters.ArcadiaUrl("Svn url for arcadia", required=True)

        with sdk2.parameters.Output:
            data_resource = sdk2.parameters.Resource("Pod agent's test_service layer", resource_type=PodAgentTestServiceLayer)

    class Context(sdk2.Task.Context):
        data_name = "pod_agent_test_service_layer.tar.gz"

    def on_execute(self):
        self.Parameters.data_resource = PodAgentTestServiceLayer(
            self, self.Parameters.description, self.Context.data_name, ttl=90
        )

        binary_path = self.build()

        layer_folder = os.path.dirname(os.path.dirname(binary_path))
        binary_name = os.path.basename(binary_path)
        layer_path = "{}.tar.gz".format(os.path.join(layer_folder, binary_name))

        logging.info("create {} archive...".format(layer_path))
        subprocess.check_call(["tar", "-zcvf", layer_path, "-C", layer_folder, binary_name])
        logging.info("archive created")

        os.rename(layer_path, str(self.Parameters.data_resource.path))

    def build(self):
        target = PodAgentTestServiceLayer.arcadia_build_path
        build_dir = tempfile.mkdtemp(dir=os.getcwd())

        with get_arcadia(self.Parameters.arcadia_url) as arcadia:
            logging.info("build pod_agent_test_service binary...")
            sdk.do_build(
                consts.YMAKE_BUILD_SYSTEM,
                source_root=arcadia,
                build_type=consts.RELEASE_BUILD_TYPE,
                targets=[target],
                results_dir=build_dir,
                clear_build=False,
            )
            logging.info("pod_agent_test_service built")

        return os.path.join(build_dir, target)
