import logging
import os
import subprocess
import tempfile
import yaml

from sandbox import sdk2
import sandbox.common.types.task as ctt
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.projects.common import apihelpers
from sandbox.projects.common import constants as consts
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.infra.BuildPodAgentBinary import BuildPodAgentBinary
from sandbox.projects.infra.CreatePodAgentTestServiceLayer import CreatePodAgentTestServiceLayer
from sandbox.projects.infra.common import get_arcadia
from sandbox.projects.infra.resources import PodAgentBinary
from sandbox.projects.infra.resources import PodAgentTestServiceLayer


DEFAULT_CHECK_TRIES = 50
DEFAULT_CHECK_INTERVAL = 20
DEFAULT_VCPU_LIMIT = 100


class ReleasePodAgentWithTestServiceOnStage(sdk2.Task):
    """
        Release pod agent and test service on stage (https://deploy.yandex-team.ru or https://test.deploy.yandex-team.ru)
    """

    class Context(sdk2.Task.Context):
        build_pod_agent_task_id = ""
        create_test_service_layer_task_id = ""
        spec_file = None
        cluster = None
        per_cluster_settings = None
        check_tries = None
        check_interval = None
        vcpu_limit = None

    def on_execute(self):
        if self.Context.spec_file is None:
            raise Exception("spec_file not provided in task context")

        if self.Context.cluster is None:
            raise Exception("cluster not provided in task context")

        if self.Context.per_cluster_settings is None:
            raise Exception("per_cluster_settings not provided in task context")

        self.prepare_resources()

        pod_agent_binary = apihelpers.list_task_resources(self.Context.build_pod_agent_task_id, resource_type=PodAgentBinary, limit=1)[0]
        test_service_layer = apihelpers.list_task_resources(self.Context.create_test_service_layer_task_id, resource_type=PodAgentTestServiceLayer, limit=1)[0]

        spec = Arcadia.cat("arcadia:/arc/trunk/arcadia/infra/pod_agent/tools/stage_deploy/spec_templates/{}".format(self.Context.spec_file))

        spec = spec.replace("CLUSTER_TEMPLATE", self.Context.cluster)

        spec = spec.replace("PER_CLUSTER_SETTINGS_TEMPLATE", self.gen_per_cluster_settings(self.Context.per_cluster_settings))

        spec = spec.replace("POD_AGENT_BINARY_URL_TEMPLATE", pod_agent_binary.proxy_url)
        spec = spec.replace("POD_AGENT_BINARY_CHECKSUM_TEMPLATE", "MD5:{}".format(pod_agent_binary.file_md5))

        spec = spec.replace("TEST_SERVICE_URL_TEMPLATE", test_service_layer.proxy_url)
        spec = spec.replace("TEST_SERVICE_CHECKSUM_TEMPLATE", "MD5:{}".format(test_service_layer.file_md5))

        spec = spec.replace("VCPU_LIMIT_TEMPLATE", str(self.Context.vcpu_limit or DEFAULT_VCPU_LIMIT))

        logging.info("spec.yml:\n{}".format(spec))

        stage_id = yaml.load(spec, Loader=yaml.SafeLoader)["meta"]["id"]

        dctl_path = self.build_tool("infra/dctl/bin", "dctl")
        dctl_dir = os.path.dirname(dctl_path)
        stage_waiter_path = self.build_tool("infra/pod_agent/tools/stage_deploy/stage_waiter", "stage-waiter")

        spec_path = os.path.join(dctl_dir, "spec.yml")

        with open(spec_path, "w") as f:
            f.write(spec)

        env = os.environ.copy()
        env.update({"YP_TOKEN": sdk2.Vault.data("robot-deploy-test", "yp-token")})
        env.update({"DCTL_YP_TOKEN": sdk2.Vault.data("robot-deploy-test", "yp-token")})

        # dctl use getpass.getuser() to get user
        env.update({"LOGNAME": "robot-deploy-test"})

        logging.info("run ./dctl put stage spec.yml -c {}".format(self.Context.cluster))
        subprocess.check_call([
            dctl_path,
            "put", "stage",
            spec_path,
            "-c", self.Context.cluster,
        ], env=env)

        check_tries = self.Context.check_tries or DEFAULT_CHECK_TRIES
        check_interval = self.Context.check_interval or DEFAULT_CHECK_INTERVAL
        logging.info("run ./stage_waiter {} {} --tries {} --interval {}".format(self.Context.cluster, stage_id, check_tries, check_interval))

        subprocess.check_call([
            stage_waiter_path,
            self.Context.cluster,
            stage_id,
            "--tries", str(check_tries),
            "--interval", str(check_interval),
        ], env=env)

        logging.info("realeased")

    def gen_per_cluster_settings(self, per_cluster_settings):
        per_cluster_settings = per_cluster_settings.split(";")

        full_settings = []
        for cluster_settings in per_cluster_settings:
            cluster, pod_count = cluster_settings.split(":")
            full_settings.append("{0}{1}:\n{2}pod_count: {3}".format((" " * 10 if cluster_settings != per_cluster_settings[0] else ""), cluster, " " * 12, pod_count))

        return "\n".join(full_settings)

    def build_tool(self, target, tool):
        build_dir = tempfile.mkdtemp()

        with get_arcadia(Arcadia.ARCADIA_TRUNK_URL) as arcadia:
            logging.info("build {}".format(target))
            sdk.do_build(
                consts.YMAKE_BUILD_SYSTEM,
                source_root=arcadia,
                build_type=consts.RELEASE_BUILD_TYPE,
                targets=[target],
                results_dir=build_dir,
                clear_build=False,
            )
            logging.info("{} built".format(target))

        return os.path.join(build_dir, target, tool)

    def prepare_resources(self):
        with self.memoize_stage.prepare_resources:
            build_pod_agent_task = BuildPodAgentBinary(
                self,
                description="build pod_agent binary for realease on stage",
                checkout_arcadia_from_url=Arcadia.ARCADIA_TRUNK_URL,
                priority=ctt.Priority(ctt.Priority.Class.SERVICE, ctt.Priority.Subclass.HIGH),
                test=True,
            )
            create_test_service_layer_task = CreatePodAgentTestServiceLayer(
                self,
                description="create pod_agent_test_serivce layer for realease on stage",
                checkout_arcadia_from_url=Arcadia.ARCADIA_TRUNK_URL,
                priority=ctt.Priority(ctt.Priority.Class.SERVICE, ctt.Priority.Subclass.HIGH),
            )

            build_pod_agent_task.enqueue()
            create_test_service_layer_task.enqueue()

            self.Context.build_pod_agent_task_id = build_pod_agent_task.id
            self.Context.create_test_service_layer_task_id = create_test_service_layer_task.id

            self.Context.save()

            raise sdk2.WaitTask([build_pod_agent_task.id, create_test_service_layer_task.id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)
