import json
import logging
import os
import subprocess
import requests
import tempfile
import time

from sandbox import sdk2
from sandbox.common import errors
from sandbox.projects.common.binary_task import deprecated as binary_task
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common import network


CLUSTERS = [
    # (cluster_name, sync_cycle_time, number_of_shards)
    # one day MAN will return
    ("sas-test", 30, 5),
    ("man-pre", 30, 5),
    ("sas", 200, 5),
#    ("man", 200, 5),
    ("vla", 200, 5),
    ("myt", 50, 5),
    ("iva", 50, 5),
    ("xdc", 100, 5),
]


class DiffServiceController(sdk2.Resource):
    """
        Endpoints diff
    """


class ServiceControllerDiffTest(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """
        Compare builded endpoints with poduction service controller
        If run is maual, uses last SERVICE_CONTROLLER_BINARY resource to compare with prod
    """

    class Parameters(sdk2.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)
        service_controller_testing_binary = sdk2.parameters.Integer("Binary resource id", required=True)

    class Context(sdk2.Task.Context):
        service_controller_prod_binaries = []

    def on_save(self):
        super(ServiceControllerDiffTest, self).on_save()

    def on_execute(self):
        super(ServiceControllerDiffTest, self).on_execute()
        self._execute()

    def _execute(self):
        from yp.client import YpClient

        self.binaries = {}

        self.env = os.environ.copy()
        self.env.update({"YP_TOKEN": sdk2.Vault.data("robot-test-srvc-ctrl", "yp-token")})
        self.env.update({"YT_TOKEN": sdk2.Vault.data("robot-test-srvc-ctrl", "yt-token")})

        nanny_client = nanny.NannyClient(
            api_url="http://nanny.yandex-team.ru/",
            oauth_token=sdk2.Vault.data("robot-test-srvc-ctrl", "nanny-token")
        )

        diff_new = {}
        diff_old = {}
        for cluster, timeout, number_of_shards in CLUSTERS:
            self.env.update({"YP_CLUSTER": cluster})
            logging.info("Begin test on {} cluster".format(cluster))
            self.Context.service_controller_prod_binaries.append({cluster: self.get_relev_binary(nanny_client, cluster)})
            self.Context.save()

            yp_client = YpClient(address="{}.yp.yandex.net:8090".format(cluster), config=dict(token=self.env["YP_TOKEN"]))
            timestamp = yp_client.generate_timestamp()
            logging.info("Timestamp: {}".format(timestamp))

            diff_new.update(self.get_diff(self.Parameters.service_controller_testing_binary, cluster, timeout, number_of_shards, timestamp))
            diff_old.update(self.get_diff(self.Context.service_controller_prod_binaries[-1][cluster], cluster, timeout, number_of_shards, timestamp))

        diff, ok = self.check_diffs(diff_new, diff_old)
        resource = DiffServiceController(self, "Endpoints diff by clusters.", "diff", ttl=60)
        resource_data = sdk2.ResourceData(resource)
        diff_path = os.path.join(tempfile.mkdtemp(dir=os.getcwd()), "diff")
        with open(diff_path, "w") as f:
            f.write(json.dumps({"new": diff_new, "old": diff_old, "new_vs_old": diff}, indent=4))
        os.rename(diff_path, str(resource_data.path))
        resource_data.ready()

        if not ok:
            raise errors.TaskError("Diff is not empty. See resource with diff for more information")

    def check_diffs(self, diff_new, diff_old):
        diff = {}
        ok = True
        for cluster, _, _ in CLUSTERS:
            def fill_empty(diff, cluster, key, content):
                logging.info("before" + str(diff))
                if key not in diff[cluster]:
                    diff[cluster].update({key: content})
                logging.info("after" + str(diff))

            fill_empty(diff_new, cluster, "insert", {"endpoints": []})
            fill_empty(diff_old, cluster, "insert", {"endpoints": []})
            fill_empty(diff_new, cluster, "remove", {"ids": []})
            fill_empty(diff_old, cluster, "remove", {"ids": []})

            insert_new = set(diff_new[cluster]["insert"]["endpoints"])
            insert_old = set(diff_old[cluster]["insert"]["endpoints"])
            remove_new = set(diff_new[cluster]["remove"]["ids"])
            remove_old = set(diff_new[cluster]["remove"]["ids"])

            diff.update(
                {
                    cluster: {
                        "insert": {
                            "+++": list(insert_new - insert_old),
                            "---": list(insert_old - insert_new),
                        },
                        "remove": {
                            "+++": list(remove_new - remove_old),
                            "---": list(remove_old - remove_new)
                        }
                    }
                }
            )

            ok = ok and (insert_new == insert_old) and (remove_new == remove_old)

        return diff, ok

    def get_diff(self, resource_id, cluster, timeout, number_of_shards, timestamp):
        import yt.wrapper as yt

        lock_dir = "//tmp/service_controller_test/leader_{}".format(time.time())

        yt_proxy = "yp-{}".format(cluster)
        args = [
            "-V", "Controller.Controller.SyncInterval=30m",
            "-V", "Controller.Controller.ThreadPoolSize=5",
            "-V", "Controller.LeadingInvader.Path={}".format(lock_dir),
            "-V", "Controller.LeadingInvader.Proxy={}".format(yt_proxy),
            "-V", "Controller.LeadingInvader.RetryAcquireLockInterval=5s",
            "-V", "Controller.LeadingInvader.Timeout=10m",
            "-V", "Controller.YpClient.Address={}.yp.yandex.net:8090".format(cluster),
            "-V", "Controller.YpClient.ReadOnlyMode=true",
            "-V", "Controller.YpClient.SnapshotTimestamp={}".format(timestamp),
            "-V", "Controller.YpClient.ThreadPoolSize=5",
            "-V", "ShardsConfig.EnsureShardMasterLivenessTimeout=0s",
            "-V", "ShardsConfig.LivenessLeadingInvader.Proxy={}".format(yt_proxy),
            "-V", "ShardsConfig.ManagedByMaster=false",
        ]

        logging.info("Creating map node '{}'".format(lock_dir))
        yt_client = yt.YtClient(proxy=yt_proxy, token=self.env["YT_TOKEN"])
        with yt_client.Transaction():
            try:
                if not yt_client.exists(lock_dir):
                    for shard_id in range(number_of_shards):
                        yt_client.create("map_node", lock_dir + ("_shard_id_" + str(shard_id) if number_of_shards > 1 else ""), recursive=True, ignore_existing=True)
                    yt_client.create("map_node", lock_dir, recursive=True, ignore_existing=True)
            except yt.YtResponseError as err:
                logging.info("Failed to create map node '{}'. Error: '{}'".format(lock_dir, str(err).replace("\n", "\\n")))
        logging.info("Map node created")

        logging.info("run service_controller from resource #{}".format(resource_id))
        log_path = os.path.join(tempfile.mkdtemp(dir=os.getcwd()), "eventlog")
        bin_path = self.run_from_resource(resource_id, args + ["-V", "Controller.Logger.Path={}".format(log_path)], timeout)
        logging.info(subprocess.check_output([bin_path, "print_log", "-r", log_path], stderr=subprocess.STDOUT))

        diff_str = subprocess.check_output([bin_path, "print_diff", log_path])
        logging.info("diff:\n{}".format(diff_str))

        return {cluster: json.loads(diff_str)}

    def get_relev_binary(self, client, cluster):
        runtime_attrs = client.get_service_runtime_attrs("{}_yp_service_controller".format(cluster.replace("-", "_")))
        for it in runtime_attrs["content"]["resources"]["sandbox_files"]:
            if it["resource_type"] == "SERVICE_CONTROLLER_BINARY":
                return it["resource_id"]

        raise ValueError("Can't find binary in runtime_attrs of '{}' cluster.".format(cluster))

    def download_sandbox_resource(self, resource_id):
        local_path = tempfile.mkdtemp(dir=os.getcwd())
        logging.info("download sandbox resource with id={id} to local_path={path}".format(id=resource_id, path=local_path))

        rsp = requests.get("https://sandbox.yandex-team.ru:443/api/v1.0/resource/{}".format(resource_id))
        rsp.raise_for_status()

        resource = rsp.json()
        subprocess.check_call(["sky", "get", "--dir={}".format(local_path), resource["skynet_id"]])
        logging.info("download complete")

        files = subprocess.check_output(["sky", "files", "--json", resource["skynet_id"]])
        return os.path.join(local_path, json.loads(files)[0]["name"])

    def run_binary(self, binary_path, args, duration):
        port = network.get_free_port()
        process = subprocess.Popen([binary_path, "run"] + ["-V", "Controller.HttpService.Port={}".format(port)] + args, stdout=subprocess.PIPE, stderr=subprocess.PIPE, env=self.env)

        time.sleep(duration)  # wait for sync
        process.kill()

    def run_from_resource(self, resource_id, args, duration):
        binary_path = ""
        if resource_id in self.binaries:
            binary_path = self.binaries[resource_id]
        else:
            binary_path = self.download_sandbox_resource(resource_id)
            self.binaries.update({resource_id: binary_path})

        self.run_binary(binary_path, args, duration)
        return binary_path
