import json
import logging
import os
import subprocess
import requests
import tempfile
import time

from sandbox import sdk2
from sandbox.common import errors
from sandbox.projects.common import apihelpers
from sandbox.projects.common.binary_task import deprecated as binary_task
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common import network
from sandbox.projects.infra.resources import YP_CAUTH_EXPORT_PACKAGE


CLUSTERS = {
    "sas-test",
    "man-pre",
    "sas",
    "man",
    "vla",
    "myt",
    "iva",
    "xdc",
}


class NannyServiceInfo:
    def __init__(self, resource_id, config):
        self.resource_id = resource_id
        self.config = config


class DiffYpCauthExport(sdk2.Resource):
    """
        Domain and owners info diff
    """


class YpCauthExportDiffTest(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """
        Compare responses of /api/cauth/domain and /api/cauth/owners routes with production service
    """

    class Parameters(sdk2.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)

    class Context(sdk2.Task.Context):
        yp_cauth_export_testing_package = None
        yp_cauth_export_prod_packages = []

    def on_save(self):
        super(YpCauthExportDiffTest, self).on_save()

    def on_execute(self):
        super(YpCauthExportDiffTest, self).on_execute()
        self._execute()

    def _execute(self):
        from yp.client import YpClient

        if not self.Context.yp_cauth_export_testing_package:
            self.Context.yp_cauth_export_testing_package = apihelpers.get_last_resource(YP_CAUTH_EXPORT_PACKAGE).id
            self.Context.save()
            logging.info("yp_cauth_export_testing_package is not provided from parent task, using last 'YP_CAUTH_EXPORT_PACKAGE' sandbox resource")

        self.binaries = {}

        self.env = os.environ.copy()
        self.env.update({"YT_TOKEN": sdk2.Vault.data("robot-yp-cauth", "yt-token")})
        self.env.update({"YP_TOKEN": sdk2.Vault.data("robot-yp-cauth", "yp-token")})

        nanny_client = nanny.NannyClient(
            api_url="http://nanny.yandex-team.ru/",
            oauth_token=sdk2.Vault.data("robot-yp-cauth", "nanny-token")
        )

        res_new = {}
        res_old = {}
        for cluster in CLUSTERS:
            self.env.update({"YP_CLUSTER": cluster})
            logging.info("Begin test on {} cluster".format(cluster))

            nanny_service_info = self.get_nanny_service_info(nanny_client, cluster)

            config_path = os.path.join(tempfile.mkdtemp(dir=os.getcwd()), "config.json")
            with open(config_path, "w") as f:
                f.write(nanny_service_info.config)

            self.Context.yp_cauth_export_prod_packages.append({cluster: nanny_service_info.resource_id})
            self.Context.save()

            yp_client = YpClient(address="{}.yp.yandex.net:8090".format(cluster), config=dict(token=self.env["YP_TOKEN"]))
            timestamp = yp_client.generate_timestamp()
            logging.info("Timestamp: {}".format(timestamp))

            res_new.update(self.get_domain_info(self.Context.yp_cauth_export_testing_package, cluster, timestamp, config_path))
            res_old.update(self.get_domain_info(self.Context.yp_cauth_export_prod_packages[-1][cluster], cluster, timestamp, config_path))

        diff, ok = self.check_results(res_new, res_old)
        resource = DiffYpCauthExport(self, "Domain and owners info diff by clusters.", "diff", ttl=30)
        resource_data = sdk2.ResourceData(resource)
        diff_path = os.path.join(tempfile.mkdtemp(dir=os.getcwd()), "diff")
        with open(diff_path, "w") as f:
            f.write(json.dumps({"new": res_new, "old": res_old, "new_vs_old": diff}, indent=4))
        os.rename(diff_path, str(resource_data.path))
        resource_data.ready()

        if not ok:
            raise errors.TaskError("Diff is not empty. See resource with diff for more information")

    def check_results(self, res_new, res_old):
        import xmltodict

        logging.info("Building diff...")
        diff = {"owners": {}, "domain": {}}
        ok = True
        for cluster in CLUSTERS:
            logging.info("Cluster: {}".format(cluster))
            logging.info("Raw new owners: {}".format(res_new[cluster]["owners"]))
            logging.info("Raw old owners: {}".format(res_old[cluster]["owners"]))

            logging.info("Raw new domain: {}".format(res_new[cluster]["domain"]))
            logging.info("Raw old domain: {}".format(res_old[cluster]["domain"]))

            diff["owners"].update({
                cluster: {
                    "added": [],
                    "deleted": [],
                    "changed": {},
                }
            })

            diff["domain"].update({
                cluster: {
                    "message": "",
                    "group": {
                        "added": [],
                        "deleted": [],
                        "changed": {},
                    }
                }
            })

            def parse_owners(owners):
                res = {}
                for i in owners.splitlines():
                    splitted = i.split()
                    group_id, users = splitted[0], splitted[1].split(",")
                    if group_id in res:
                        res[group_id] += users
                    else:
                        res[group_id] = users

                #  sort users
                for users in res.itervalues():
                    users.sort()

                return res

            #  build /owners diff
            new_owners = parse_owners(res_new[cluster]["owners"])
            old_owners = parse_owners(res_old[cluster]["owners"])

            for group_id, users in new_owners.iteritems():
                if group_id not in old_owners:
                    ok = False
                    diff["owners"][cluster]["added"].append((group_id, users))
                elif old_owners[group_id] != users:
                    ok = False
                    diff["owners"][cluster]["changed"].update({
                        group_id: {
                            "from": old_owners[group_id],
                            "to": users
                        }
                    })

            for group_id, users in old_owners.iteritems():
                if group_id not in new_owners:
                    ok = False
                    diff["owners"][cluster]["deleted"].append((group_id, users))

            def parse_xml(xml_str):
                xml_dict = xmltodict.parse(xml_str)
                domain_name = xml_dict["domain"]["@name"]
                res = {}

                groups = xml_dict["domain"].get("group", [])
                if type(groups) is not list:
                    groups = [groups]
                for group in groups:
                    host = group.get("host", None)
                    if not host:
                        continue

                    res.update({
                        group["@name"]: {
                            host["@name"]: {
                                "type": host["type"],
                                "macros": host["macroses"]["macros"]["@name"],
                                "users": sorted([user["@login"] for user in host.get("users", {}).get("user", [])]),
                            }
                        }
                    })

                return domain_name, res

            #  build /domain diff
            new_domain_name, new_domain = parse_xml(res_new[cluster]["domain"])
            old_domain_name, old_domain = parse_xml(res_old[cluster]["domain"])

            #  contains only one key which is equal to domain name
            if new_domain_name != old_domain_name:
                ok = False
                diff["domain"][cluster]["message"] = "Domain names are different"

            for group_id, info in new_domain.iteritems():
                if group_id not in old_domain:
                    diff["domain"]["group"]["added"].append(group_id)
                elif info != old_domain[group_id]:
                    diff["domain"]["group"]["changed"].update({
                        group_id: {
                            "from": old_domain[group_id],
                            "to": info,
                        }
                    })

            for group_id in old_domain:
                if group_id not in new_domain:
                    diff["domain"]["group"]["deleted"].append(group_id)

        return diff, ok

    def get_domain_info(self, resource_id, cluster, timestamp, config_path):
        import yt.wrapper as yt

        lock_dir = "//tmp/yp_cauth_export_test/leader_{}".format(time.time())

        yt_proxy = "yp-{}".format(cluster)
        args = [
            "-V", "Controller.LeadingInvader.Path={}".format(lock_dir),
            "-V", "Controller.LeadingInvader.Proxy={}".format(yt_proxy),
            "-V", "Controller.LeadingInvader.Timeout=10m",
            "-V", "Controller.YpClient.Address={}.yp.yandex.net:8090".format(cluster),
            "-V", "Controller.YpClient.ReadOnlyMode=true",
            "-V", "Controller.YpClient.SnapshotTimestamp={}".format(timestamp),
            "-V", "CAuthClient.ReadOnlyMode=true",
            "-V", "CAuthUpdater.DomainName=yp-{}".format(cluster),
            "-V", "RequestsStorage.Mock=true",
            "--config", config_path,
        ]

        logging.info("Creating map node '{}'".format(lock_dir))
        yt_client = yt.YtClient(proxy=yt_proxy, token=self.env["YT_TOKEN"])
        with yt_client.Transaction():
            try:
                if not yt_client.exists(lock_dir):
                    yt_client.create("map_node", lock_dir, recursive=True, ignore_existing=True)
            except yt.YtResponseError as err:
                logging.info("Failed to create map node '{}'. Error: '{}'".format(lock_dir, str(err).replace("\n", "\\n")))
        logging.info("Map node created")

        logging.info("run yp_cauth_export from resource #{}".format(resource_id))
        log_path = os.path.join(tempfile.mkdtemp(dir=os.getcwd()), "eventlog")
        bin_path, res = self.run_from_resource(resource_id, args + ["-V", "Controller.Logger.Path={}".format(log_path)], 60)
        logging.info(subprocess.check_output([bin_path, "print_log", log_path], stderr=subprocess.STDOUT))

        return {cluster: res}

    def get_nanny_service_info(self, client, cluster):
        runtime_attrs = client.get_service_runtime_attrs("yp-cauth-export-{}".format(cluster))
        resource_id = None
        config = None
        for it in runtime_attrs["content"]["resources"]["sandbox_files"]:
            if it["resource_type"] == "YP_CAUTH_EXPORT_PACKAGE":
                resource_id = it["resource_id"]
                break

        if not resource_id:
            raise ValueError("Can't find package in runtime_attrs of '{}' cluster.".format(cluster))

        for it in runtime_attrs["content"]["resources"]["static_files"]:
            if it["local_path"] == "config.json":
                config = it["content"]
                break

        if not config:
            raise ValueError("Can't find config in runtime_attrs of '{}' cluster.".format(cluster))

        return NannyServiceInfo(resource_id, config)

    def download_sandbox_resource(self, resource_id):
        local_path = tempfile.mkdtemp(dir=os.getcwd())
        logging.info("download sandbox resource with id={id} to local_path={path}".format(id=resource_id, path=local_path))

        rsp = requests.get("https://sandbox.yandex-team.ru:443/api/v1.0/resource/{}".format(resource_id))
        rsp.raise_for_status()

        resource = rsp.json()
        subprocess.check_call(["sky", "get", "--dir={}".format(local_path), resource["skynet_id"]])
        logging.info("download complete")

        files = subprocess.check_output(["sky", "files", "--json", resource["skynet_id"]])
        return os.path.join(local_path, json.loads(files)[0]["name"])

    def unpack_package(self, local_path):
        logging.info("Unpacking package from {}".format(local_path))
        out_dir = tempfile.mkdtemp(dir=os.getcwd())

        subprocess.check_call(["tar", "-xf", local_path, "-C", out_dir])
        logging.info("Unpacking complete")

        return os.path.join(out_dir, "yp_cauth_export/yp_cauth_export")

    def run_binary(self, binary_path, args, duration):
        port = network.get_free_port()
        process = subprocess.Popen([binary_path, "run"] + ["-V", "Controller.HttpService.Port={}".format(port)] + args, stdout=subprocess.PIPE, stderr=subprocess.PIPE, env=self.env)

        time.sleep(duration)  # wait for sync

        def get_by_route(route):
            rsp = requests.get("http://localhost:{}/{}".format(port, route.lstrip("/")))
            rsp.raise_for_status()
            return rsp.content

        domain = get_by_route("/api/cauth/domain")
        owners = get_by_route("/api/cauth/owners")

        process.kill()

        return {"domain": domain, "owners": owners}

    def run_from_resource(self, resource_id, args, duration):
        binary_path = ""
        if resource_id in self.binaries:
            binary_path = self.binaries[resource_id]
        else:
            archive_path = self.download_sandbox_resource(resource_id)
            binary_path = self.unpack_package(archive_path)
            self.binaries.update({resource_id: binary_path})

        return (binary_path, self.run_binary(binary_path, args, duration))
