import json
import logging
import os
import subprocess
import requests
import re
import tempfile
import time

from sandbox import sdk2
from sandbox.common import errors

from sandbox.projects.common import apihelpers
from sandbox.projects.common import constants as sdk_constants
from sandbox.projects.common import network
from sandbox.projects.common.arcadia import sdk as arcadiasdk
from sandbox.projects.common.binary_task import deprecated as binary_task
from sandbox.projects.common.nanny import nanny

from sandbox.sdk2.vcs.svn import Arcadia

from sandbox.projects.infra.common import get_arcadia
from sandbox.projects.infra.resources import YP_EXPORT_PACKAGE, YP_EXPORT_SOURCES_CACHING_PROXY_PACKAGE


def get_latest_gencfg_tag():
    tags = requests.get("https://api.gencfg.yandex-team.ru/trunk/tags").json()["tags"]
    res = (0, 0)

    for tag in tags:
        major = int(tag.split("-")[1])
        minor = int(tag.split("-")[2][1:])
        res = max(res, (major, minor))

    return "stable-{}-r{}".format(*res)


CLUSTERS = {
    "sas-test",
    "man-pre",
    "sas",
    "man",
    "vla",
    "myt",
    "iva",
    "xdc",
}

SLEEP_TIME = 120
CACHING_PROXY_SERVER_NAME = 'http://localhost'
CACHING_PROXY_BINARY_NAME = 'sources_caching_proxy'


class NannyServiceInfo:
    def __init__(self, resource_id, config, group_mapping, project_mapping, proxy_port, save_on_disk=True):
        self.resource_id = resource_id
        self.config = config
        self.group_mapping = group_mapping
        self.project_mapping = project_mapping
        self.proxy_port = proxy_port

        self._set_proxy_http_paths()
        self._set_gencfg_tag()
        if save_on_disk:
            self._write_static_resources_to_files()

    def _write_static_resources_to_files(self):
        self.static_resources_dir = tempfile.mkdtemp(dir=os.getcwd())

        self.config_path = os.path.join(self.static_resources_dir, "config.json")
        with open(self.config_path, "w") as f:
            f.write(self.config)

        self.group_mapping_path = os.path.join(self.static_resources_dir, "group_mapping.json")
        with open(self.group_mapping_path, "w") as f:
            f.write(self.group_mapping)

        self.project_mapping_path = os.path.join(self.static_resources_dir, "project_mapping.json")
        with open(self.project_mapping_path, "w") as f:
            f.write(self.project_mapping)

    def _set_proxy_http_paths(self):
        config = json.loads(self.config)
        for component, config_values in config.items():
            if "Http" in config_values:
                http = config_values['Http']
                full_url_without_schema = re.sub('https://', '', http.get('Host', '')) + http.get('ApiPrefix', '')

                http['Host'] = CACHING_PROXY_SERVER_NAME
                http['ApiPrefix'] = '/source/' + full_url_without_schema
                http['Port'] = self.proxy_port

        self.config = json.dumps(config, indent=4)

    def _set_gencfg_tag(self):
        config = json.loads(self.config)
        if not config['Gencfg'].get('Tag'):
            config['Gencfg']['Tag'] = get_latest_gencfg_tag()

        self.config = json.dumps(config)


class DiffYpExport(sdk2.Resource):
    """
        YP objects diff
    """


class YpExportDiffTest(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """
        Compare builded binary with production yp_export
        If run is maual, uses last YP_EXPORT_PACKAGE resource to compare with prod
    """

    class Parameters(sdk2.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)

    class Context(sdk2.Task.Context):
        yp_export_testing_package = None
        yp_export_prod_packages = []

    def on_save(self):
        super(YpExportDiffTest, self).on_save()

    def on_execute(self):
        super(YpExportDiffTest, self).on_execute()
        self._execute()

    def _execute(self):
        from yp.client import YpClient

        if not self.Context.yp_export_testing_package:
            self.Context.yp_export_testing_package = apihelpers.get_last_resource(YP_EXPORT_PACKAGE).id
            self.Context.save()
            logging.info(
                "yp_export_testing_package is not provided from parent task, "
                "using last 'YP_EXPORT_PACKAGE' sandbox resource"
            )

        proxy_binary_path = self.build_proxy()
        proxy_port = network.get_free_port()
        proxy_proc = self.run_proxy(proxy_binary_path, ["--port", str(proxy_port), "--retry", str(10)])

        self.binaries = {}

        self.env = os.environ.copy()
        self.env.update({"YP_TOKEN": sdk2.Vault.data("robot-yp-export", "yp-token")})
        self.env.update({"YT_TOKEN": sdk2.Vault.data("robot-yp-export", "yt-token")})
        self.env.update({"ARCANUM_TOKEN": sdk2.Vault.data("robot-yp-export", "arcanum-token")})

        nanny_client = nanny.NannyClient(
            api_url="http://nanny.yandex-team.ru/",
            oauth_token=sdk2.Vault.data("robot-yp-export", "nanny-token")
        )

        all_diff_new = {}
        all_diff_old = {}
        for cluster in CLUSTERS:
            self.env.update({"YP_CLUSTER": cluster})
            logging.info("Starting test on {} cluster...".format(cluster))
            nanny_service_info = self.get_nanny_service_info(nanny_client, cluster, proxy_port)

            logging.info(
                'Using the following config for {} cluster:\n{}'.format(
                    cluster,
                    subprocess.check_output(["cat", nanny_service_info.config_path], stderr=subprocess.STDOUT)
                )
            )

            self.Context.yp_export_prod_packages.append({cluster: nanny_service_info.resource_id})
            self.Context.save()

            yp_client = YpClient(
                address="{}.yp.yandex.net:8090".format(cluster),
                config=dict(token=self.env["YP_TOKEN"])
            )
            timestamp = yp_client.generate_timestamp()
            logging.info("Using timestamp: {}".format(timestamp))

            all_diff_new.update(
                self.get_diff(self.Context.yp_export_testing_package, cluster, timestamp, nanny_service_info)
            )
            all_diff_old.update(
                self.get_diff(self.Context.yp_export_prod_packages[-1][cluster], cluster, timestamp, nanny_service_info)
            )

        summary_diff, ok = self.check_diffs(all_diff_new, all_diff_old)
        resource = DiffYpExport(self, "YP objects diff by clusters.", "diff", ttl=30)
        resource_data = sdk2.ResourceData(resource)
        diff_path = os.path.join(tempfile.mkdtemp(dir=os.getcwd()), "diff")
        with open(diff_path, "w") as f:
            f.write(json.dumps({"new": all_diff_new, "old": all_diff_old, "new_vs_old": summary_diff}, indent=4))
        os.rename(diff_path, str(resource_data.path))
        resource_data.ready()

        proxy_proc.kill()

        if not ok:
            raise errors.TaskError("Diff is not empty. See resource with diff for more information")

    def check_diffs(self, all_diff_new, all_diff_old):
        from deepdiff import DeepDiff

        summary_diff = {}
        ok = True
        for cluster in CLUSTERS:
            old_cluster_diff = all_diff_old[cluster].copy()
            new_cluster_diff = all_diff_new[cluster].copy()

            # because of possible retries
            del old_cluster_diff['sync_cycles']
            del new_cluster_diff['sync_cycles']

            cluster_requests_diff = DeepDiff(old_cluster_diff, new_cluster_diff, verbose_level=2, ignore_order=True)
            ok = ok and len(cluster_requests_diff) == 0
            summary_diff.update({cluster: cluster_requests_diff})

        return summary_diff, ok

    def get_diff(self, resource_id, cluster, timestamp, nanny_service_info):
        import yt.wrapper as yt

        lock_dir = "//tmp/yp_export_test/leader_{}".format(time.time())
        port = network.get_free_port()
        log_path = os.path.join(tempfile.mkdtemp(dir=os.getcwd()), "eventlog")

        yt_proxy = "yp-{}".format(cluster)
        args = [
            "--config", nanny_service_info.config_path,
            "-V", "Controller.Controller.SyncInterval=30m",
            "-V", "Controller.LeadingInvader.Path={}".format(lock_dir),
            "-V", "Controller.LeadingInvader.Proxy={}".format(yt_proxy),
            "-V", "Controller.LeadingInvader.RetryAcquireLockInterval=5s",
            "-V", "Controller.LeadingInvader.Timeout=10m",
            "-V", "Controller.YpClient.Address={}.yp.yandex.net:8090".format(cluster),
            "-V", "Controller.YpClient.ReadOnlyMode=true",
            "-V", "Controller.YpClient.SnapshotTimestamp={}".format(timestamp),
            "-V", "Controller.HttpService.Port={}".format(port),
            "-V", "Controller.Logger.Path={}".format(log_path),
            "-V", "Staff.Http.AuthToken={}".format(self.env["YT_TOKEN"]),
            "-V", "Abc.Http.AuthToken={}".format(self.env["YT_TOKEN"]),
            "-V", "Cpu.Http.AuthToken={}".format(self.env["ARCANUM_TOKEN"]),
            "-V", "Dispenser.Http.AuthToken={}".format(self.env["YT_TOKEN"]),
            "-V", "Walle.HostFqdnProvider.Project2SegmentMapPath={}".format(nanny_service_info.project_mapping_path),
            "-V", "Gencfg.HostMetadataProvider.Group2SegmentMapPath={}".format(nanny_service_info.group_mapping_path),
        ]

        # until dispenser functionality is turned on on all clusters
        args += [
            "-V", "Dispenser.ExportEnabled=false"
        ]

        logging.info("Creating map node '{}'".format(lock_dir))
        yt_client = yt.YtClient(proxy=yt_proxy, token=self.env["YT_TOKEN"])
        with yt_client.Transaction():
            try:
                if not yt_client.exists(lock_dir):
                    yt_client.create("map_node", lock_dir, recursive=True, ignore_existing=True)
            except yt.YtResponseError as err:
                logging.error(
                    "Failed to create map node '{}'. Error: '{}'".format(lock_dir, str(err).replace("\n", "\\n"))
                )
                return {cluster: {}}
        logging.info("Map node created")

        logging.info("Running yp_export from resource #{} in {} cluster".format(resource_id, cluster))
        bin_path = self.run_from_resource(resource_id, args, SLEEP_TIME)

        logging.info(
            subprocess.check_output([bin_path, "print_log", "-r", log_path], stderr=subprocess.STDOUT).decode('utf-8')
        )

        diff_args = [
            "-o", "OT_NETWORK_PROJECT",
            "-o", "OT_VIRTUAL_SERVICE",
            "-o", "OT_NODE2",
            "-o", "OT_RESOURCE",
            "-o", "OT_USER",
            "-o", "OT_GROUP",
            "-o", "OT_ACCOUNT",
        ]

        diff_raw = subprocess.check_output([bin_path, "print_diff", log_path] + diff_args)

        try:
            diff_dict = json.loads(diff_raw)
            return {cluster: diff_dict}
        except json.JSONDecodeError:
            logging.error(
                "Error loading diff as json. "
                "Seems like print_diff produced unexpected result or no cycle has started yet."
            )
            return {cluster: {}}

    def get_nanny_service_info(self, client, cluster, proxy_port):
        runtime_attrs = client.get_service_runtime_attrs("yp-export-{}".format(cluster))

        resource_id = None
        config = None
        group_mapping = None
        project_mapping = None
        for it in runtime_attrs["content"]["resources"]["sandbox_files"]:
            if it["resource_type"] == "YP_EXPORT_PACKAGE":
                resource_id = it["resource_id"]
                break

        if resource_id is None:
            raise ValueError("Can't find YP_EXPORT_PACKAGE in runtime_attrs of '{}' cluster.".format(cluster))

        for it in runtime_attrs["content"]["resources"]["static_files"]:
            if it["local_path"] == "config.json":
                config = it["content"]
            elif it["local_path"] == "group_mapping.json":
                group_mapping = it["content"]
            elif it["local_path"] == "project_mapping.json":
                project_mapping = it["content"]

        if config is None or group_mapping is None or project_mapping is None:
            raise ValueError("Can't find some static resource in runtime_attrs of '{}' cluster.".format(cluster))

        return NannyServiceInfo(resource_id, config, group_mapping, project_mapping, proxy_port)

    def download_sandbox_resource(self, resource_id):
        local_path = tempfile.mkdtemp(dir=os.getcwd())
        logging.info(
            "Downloading sandbox resource with id={id} to local_path={path}".format(id=resource_id, path=local_path)
        )

        rsp = requests.get("https://sandbox.yandex-team.ru:443/api/v1.0/resource/{}".format(resource_id))
        rsp.raise_for_status()

        resource = rsp.json()
        subprocess.check_call(["sky", "get", "--dir={}".format(local_path), resource["skynet_id"]])
        logging.info("Download completed")

        files = subprocess.check_output(["sky", "files", "--json", resource["skynet_id"]])
        return os.path.join(local_path, json.loads(files)[0]["name"])

    def unpack_package(self, local_path, binary_fullname):
        logging.info("Unpacking package from {}".format(local_path))
        out_dir = tempfile.mkdtemp(dir=os.getcwd())

        subprocess.check_call(["tar", "-xf", local_path, "-C", out_dir])
        logging.info("Unpacking completed")

        return os.path.join(out_dir, binary_fullname)

    def run_binary(self, binary_path, args, duration):
        process = subprocess.Popen(
            [binary_path, "run"] + args,
            stdout=subprocess.PIPE, stderr=subprocess.PIPE, env=self.env
        )

        # wait for sync
        time.sleep(duration)
        process.kill()

    def run_from_resource(self, resource_id, args, duration):
        binary_path = ""
        if resource_id in self.binaries:
            binary_path = self.binaries[resource_id]
        else:
            archive_path = self.download_sandbox_resource(resource_id)
            binary_path = self.unpack_package(archive_path, "yp_export/yp_export")
            self.binaries[resource_id] = binary_path

        self.run_binary(binary_path, args, duration)
        return binary_path

    def run_proxy(self, path, args):
        return subprocess.Popen([path] + args, stderr=subprocess.STDOUT)

    def build_proxy(self):
        target = YP_EXPORT_SOURCES_CACHING_PROXY_PACKAGE.arcadia_build_path
        build_dir = tempfile.mkdtemp()

        with get_arcadia(Arcadia.ARCADIA_TRUNK_URL) as arcadia:
            logging.info("Building yp_export/source_caching_proxy binary...")
            arcadiasdk.do_build(
                sdk_constants.YMAKE_BUILD_SYSTEM,
                source_root=arcadia,
                targets=[target],
                results_dir=build_dir,
                clear_build=False,
            )
            logging.info("Build yp_export/source_caching_proxy completed")

        return os.path.join(build_dir, target, CACHING_PROXY_BINARY_NAME)
