import os

from sandbox import sdk2
from sandbox.common import errors
from sandbox.projects.common import binary_task
from sandbox.sdk2.helpers import subprocess

from sandbox.projects.websearch.params import ResourceWithLastReleasedValueByDefault

from sandbox.projects.infra.dns.zonefiles import resources


class GenerateDnsZoneFiles(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Parameters(sdk2.Task.Parameters):

        ext_params = binary_task.binary_release_parameters(stable=True)

        zonefiles_generator = sdk2.parameters.Resource(
            'Zone files generator binary (infra/yp_dns/tools/generate_zonefiles)',
            resource_type=resources.GenerateDnsZoneFilesBinary,
            required=True
        )

        zones_config = sdk2.parameters.JSON(
            'Zones config (by default use config compiled in generator)'
        )

        yp_token_secret = sdk2.parameters.YavSecret(
            'Secret with YP token (by default use "yp_token" key)'
        )

    def on_save(self):
        super(GenerateDnsZoneFiles, self).on_save()
        if not self.Parameters.zonefiles_generator:
            self.Parameters.zonefiles_generator = sdk2.Resource[ResourceWithLastReleasedValueByDefault(resource_type=resources.GenerateDnsZoneFilesBinary).default_value]

    def on_execute(self):
        super(GenerateDnsZoneFiles, self).on_execute()

        zonefiles_generator_binary = str(sdk2.ResourceData(self.Parameters.zonefiles_generator).path)
        result_dir = 'zonefiles'

        env = os.environ.copy()
        env['YP_TOKEN'] = self._get_secret_value(self.Parameters.yp_token_secret, 'yp_token')

        cmd = [
            zonefiles_generator_binary,
            '--output-dir', result_dir,
        ]

        with sdk2.helpers.ProcessLog(self, logger='zonefiles_generator') as pl:
            self.set_info('Run {}'.format(' '.join(cmd)))
            process = subprocess.Popen(
                cmd,
                env=env,
                stdout=pl.stdout,
                stderr=pl.stderr,
            )
            if process.wait():
                raise errors.TaskFailure()
            self.set_info('Generation successfully done')

        for zone_name in os.listdir(result_dir):
            zonefile_path = os.path.join(result_dir, zone_name)
            zonefile = resources.DnsZoneFile(
                self, '{} zonefile'.format(zone_name), zonefile_path, ttl=30,
                zone=zone_name
            )
            sdk2.ResourceData(zonefile).ready()
            self.set_info('Zonfile for zone {} created'.format(zone_name))

    def _get_secret_value(self, secret, default_key):
        key = secret.default_key if secret.default_key is not None else default_key
        return secret.data()[key]
