# -*- coding: utf-8 -*-

import json
import logging

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.common.errors import VaultError, TaskFailure

REQUIRED_FIELDS = (
    'createdAt',
    'issueWeight',
    'queue',
    'key',
    'priority',
    'resolvedAt',
    'status',
    'summary',
    'assignee',
    'bugSource',
    'spent',
    'causeOfReleaseBug',
    'start',
    'end',
    'autotesting',
    'tags',
    'components',
    'resolution',
    'sla'
)

EXPAND_FIELDS = (
    'sla',
)

REQUIRED_APPS = (
    'com.github',
    'org.reviewboard',
    'ru.yandex.arcanum',
)


class VteamIssuesData(sdk2.Resource):
    """JSON с данными тикетов из Стартрека"""
    filter_id = sdk2.Attributes.Integer("ID фильтра в Стартреке")


class VteamIssuesMeta(sdk2.Resource):
    """JSON с мета-информацией о реультате выполнения таска"""
    filter_id = sdk2.Attributes.Integer("ID фильтра в Стартреке")


class FetchStIssues(sdk2.Task):
    """Таск для выгрузки тикетов из Стартрека по номеру фильтра.

    Записывает два ресурса: один с данными тикетов, второй с мета-данными о первом ресурсе.
    Мета-данные нужны, чтобы передать информацию о первом ресурсе в кубик Нирваны, в кубик на Sandbox процессоре
    можно передать только переопределение параметров, входов нет.
    """
    _st_client_instance = None
    filter_id = None

    class Requirements(sdk2.Task.Requirements):
        environments = [PipEnvironment('startrek_client', use_wheel=True)]

    class Parameters(sdk2.Task.Parameters):
        filter_id = sdk2.parameters.Integer('ST filter', required=True)

    @property
    def _st_client(self):
        """Инициализирует и запоминает инстанс АПИ Стартрека

        :rtype: startrek_client.Startrek
        :return: инстанс клиента АПИ Стартрека
        """
        if self._st_client_instance is None:
            from startrek_client import Startrek

            try:
                token = sdk2.Vault.data('robot-serptools', 'robot-serptools_startrek_token')
            except VaultError as err:
                logging.error(err)
                raise TaskFailure("Can't obtain OAuth token for Startrek")

            self._st_client_instance = Startrek(
                token=token,
                useragent='robot-serptools'
            )
        return self._st_client_instance

    def on_execute(self):
        self.filter_id = self.Parameters.filter_id

        st_issues = self.fetch_st_issues()
        issues = [self.prepare_issue(st_issue) for st_issue in st_issues]

        data_resource_id = self.write_data(issues)
        self.write_meta(data_resource_id)

    def fetch_st_issues(self):
        """Загружает данные задач из Стартрека по фильтру. Выбирает только необходимые поля

        :rtype: list
        :return: список тикетов
        """
        logging.info('Fetching filter: {id}'.format(id=self.filter_id))

        issues = self._st_client.issues.find(filter_id=self.filter_id, fields=REQUIRED_FIELDS, expand=EXPAND_FIELDS)
        # Иногда трекер присылает дубли задач, непонятно почему. Надо убрать.
        issues = {issue.key: issue for issue in issues}.values()

        logging.info('Issues fetched: {num}'.format(num=len(issues)))

        return issues

    @staticmethod
    def prepare_issue(st_issue):
        """Дополняет сырой ответ АПИ данными и формирует сериализуемый в JSON объект

        :param st_issue: - экземпляр ответа startrek_client

        :rtype dict:
        """
        issue = st_issue.as_dict()
        # игнорируем элементы ченжлога с типом IssueCreated, потому что:
        #  1) логика работы с данными задачи была написана, когда такого типа не было; переделывать сложно, потому что:
        #  2) есть различия между задачами без истории поля совсем (там ченжлог пустой) и задачами с изменением поля,
        #     тогда у поля есть изменения None->value1 (IssueCreated) и value1->value2 (IssueUpdated/IssueWorkflow).
        #     При необходимости получить начальное значение это различие усложняет логику.
        issue['changelog'] = [change.as_dict() for change in st_issue.changelog.get_all(field=REQUIRED_FIELDS)
                              if change.type != 'IssueCreated']
        issue['remotelinks'] = [link.as_dict() for link in st_issue.remotelinks.get_all()
                                if link.object.application.type in REQUIRED_APPS]
        return issue

    def write_data(self, issues):
        """Создаёт ресурс с данными выгруженных из Стартрека задач

        :param list issues:

        :rtype int:
        :return: ID созданного ресурса
        """
        data = json.dumps(issues)

        resource = VteamIssuesData(self, "Issues JSON", "filter_data.json", filter_id=self.filter_id)

        resource_data = sdk2.ResourceData(resource)
        resource_data.path.write_bytes(data)

        return resource.id

    def write_meta(self, resource_id):
        """Создает ресурс с мета-информацией о выгруженных задачах

        :param int resource_id:
        """
        data = json.dumps({'issues_resource_id': resource_id})

        resource = VteamIssuesMeta(self, "Issues JSON Meta", "filter_meta.json", filter_id=self.filter_id)

        resource_data = sdk2.ResourceData(resource)
        resource_data.path.write_bytes(data)
