# -*- coding: utf-8 -*-

import json
import logging
import requests
from string import Template

from sandbox import sdk2
from sandbox.common.errors import TaskError, TaskFailure
from sandbox.common.types import notification as ctn


class InfraToolsLauncher(sdk2.Task):
    """Таска для запуска ручки любого сервиса и отправки результата по почте"""

    class Parameters(sdk2.Task.Parameters):
        url = sdk2.parameters.Url('Url', required=True)
        data = sdk2.parameters.String('Data to send', multiline=True, description='JSON')

        with sdk2.parameters.RadioGroup('Request type', required=True) as request_type:
            request_type.values.GET = request_type.Value('GET', default=True)
            request_type.values.POST = request_type.Value('POST')

        allow_retries = sdk2.parameters.Bool('Allow retries', default=False,
                                             description="Retries are made only if response demands it")

        with allow_retries.value[True]:
            retry_url = sdk2.parameters.Url(
                'Retry url',
                description='Use placeholder $retry for response data substitution (optional). '
                            'Leave empty to use the primary url'
            )
            retry_delay = sdk2.parameters.Integer('Retry delay (seconds)', default=120)

        send_email = sdk2.parameters.Bool('Send email', default=False)

        with send_email.value[True]:
            with sdk2.parameters.Group('Email') as email:
                recipients = sdk2.parameters.String('To')
                subject = sdk2.parameters.String('Subject')
                message = sdk2.parameters.String(
                    'Message',
                    multiline=True,
                    description='If there is any data for email in response, it will be used with higher priority'
                )
                with sdk2.parameters.RadioGroup('Type', required=True) as content_type:
                    content_type.values[ctn.Type.TEXT] = content_type.Value('Text', default=True)
                    content_type.values[ctn.Type.HTML] = content_type.Value('HTML')

    class Context(sdk2.Task.Context):
        retry_count = 0
        retry_data = None

    def on_execute(self):
        """Точка входа. При первом запуске отправляет запрос по основному урлу, при повторных - по урлу ретрая."""
        if self.Context.retry_count == 0:
            url = self.Parameters.url
        else:
            url_tmpl = self.Parameters.retry_url or self.Parameters.url
            url = Template(url_tmpl).safe_substitute(retry=self.Context.retry_data)

        response = self.send_request(url)
        self.process_response(response)

    def retry(self, retry_data):
        """Подготавливает данные для перезапроса и переводит таск в статус WAIT_TIME.

        :param str retry_data: - данные которые нужно подставить в урл для повторного запроса
        """
        logging.info('Retrying...')
        self.Context.retry_data = retry_data
        self.Context.retry_count += 1
        raise sdk2.WaitTime(self.Parameters.retry_delay)

    def send_request(self, url):
        """Отправляет запрос на основе параметров таска и возвращает ответ.

        :param str, sdk2.parameters.Url url: - URL, на который надо сделать запрос

        :return: результат запроса
        :rtype: dict

        :raises TaskError: если не получилось распарсить данные для отправки как JSON
        :raises TaskError: если не получилось распарсить ответ как JSON
        :raises TaskFailure: в случае HTTP ошибки
        """
        logging.info('Fetching <%s>...' % url)

        data = None
        if self.Parameters.data:
            try:
                data = json.loads(self.Parameters.data)
            except ValueError:
                raise TaskError('Malformed JSON in request, double check the data')

        if self.Parameters.request_type == 'GET':
            response = requests.get(url, data=data, verify=False)
        else:
            response = requests.post(url, data=data, verify=False)

        try:
            response.raise_for_status()
        except requests.HTTPError as err:
            raise TaskFailure(err)  # уводим таск в статус Failure, а не Exception

        try:
            parsed_response = response.json()
        except ValueError:
            logging.info('=== Response body start ===')
            logging.info(response.text)
            logging.info('=== Response body end ===')
            raise TaskError('Malformed JSON in response, see the log')

        return parsed_response

    def process_response(self, parsed_response):
        """Проверяет ответ на наличие ошибок, при необходимости запускает рассылку и ретраи.

        :param dict parsed_response: распаршенный JSON, вернувшийся от ручки

        :raises TaskFailure: если в ответе от ручки есть ошибка
        """
        # Данные ответа могут лежать в поле response, либо сразу в корне ответа
        response_data = parsed_response.get('response', parsed_response)

        response_error = parsed_response.get('error') or response_data.get('error')
        if response_error:
            raise TaskFailure('Internal error: %s' % response_error)

        if self.Parameters.send_email:
            email_data = response_data.get('email_data')

            # Ответ может быть списком словарей или отдельным словарём
            # Если нет данных для писем в ответе, шлём одно письмо на основе параметров таска (вызов с data=None)
            if type(email_data) is not list:
                email_data = [email_data]

            # Каждый словарь - данные для одного письма
            for email in email_data:
                self.send_email(email)

        # ретраи должны быть в конце, потому что переводят таск в статус WAIT_TIME
        if self.Parameters.allow_retries:
            retry_data = response_data.get('retry')
            if retry_data:
                self.retry(retry_data)

    def send_email(self, data=None):
        """Отправляет одно письмо на основе переданных данных. Если данных для отправки недостаточно, не делает ничего.

        :param dict or None data: данные письма с опциональными полями `to`, `subject`, `body` и `data`
        """
        if data is None:
            data = {}

        # Данные из параметров таска используются в т. ч. как значения по умолчанию, если в ответе нет какого-то поля
        recipients = data.get('to', self.Parameters.recipients)
        subject_tmpl = data.get('subject', self.Parameters.subject)
        body_tmpl = data.get('body', self.Parameters.message)
        content_type = data.get('type', self.Parameters.content_type)

        try:
            subject_tmpl = subject_tmpl.decode('utf-8')
            body_tmpl = body_tmpl.decode('utf-8')
        except UnicodeEncodeError:
            # уже в нужной кодировке
            pass

        tmpl_data = data.get('data', {})

        if not recipients or not subject_tmpl or not body_tmpl:
            logging.info('Email data is incomplete, omit sending')
            return

        if type(recipients) is not list:
            recipients = [recipients]

        subject = Template(subject_tmpl).safe_substitute(tmpl_data)
        body = Template(body_tmpl).safe_substitute(tmpl_data)

        logging.info('Sending mail')
        logging.info('To: %s' % recipients)
        logging.info('Subject: %s' % subject)
        logging.info('Message: %s' % body)
        logging.info('Type: %s' % content_type)

        self.server.notification(
            recipients=recipients,
            subject=subject.encode('utf-8'),
            body=body.encode('utf-8'),
            transport=ctn.Transport.EMAIL,
            type=content_type
        )
