# -*- coding: utf-8 -*-

from sandbox import sdk2, common
from sandbox.sandboxsdk.environments import PipEnvironment

import logging


class MoveIssueWeight(sdk2.Task):
    """Таск для переноса первого значения из истории любого поля таски Startrek в другое заданное поле.
    """
    _st_client_instance = None
    filter_id = None

    class Requirements(sdk2.Task.Requirements):
        environments = [PipEnvironment('startrek_client')]

    class Parameters(sdk2.Task.Parameters):
        filter_id = sdk2.parameters.Integer(
            'ST filter',
            required=True,
            description='Номер фильтра в Startrek'
        )
        sourceField = sdk2.parameters.String(
            'Source field',
            required=True,
            description='Поле, в котором следует искать начальное значение',
            default='storyPoints'
        )
        destinationField = sdk2.parameters.String(
            'Destination field',
            required=True,
            description='Поле, в которое следует проставить полученное значение',
            default='originalEstimate'
        )
        debugMode = sdk2.parameters.Bool(
            'Debug mode',
            required=True,
            description='Режим дебага: вместо записи значения в Startrek, выводит полученные значения в заголовок Sandbox-таски',
            default=False
        )

    @property
    def _st_client(self):
        """Инициализирует и запоминает инстанс АПИ Стартрека

        :return: инстанс клиента АПИ Стартрека
        :rtype: startrek_client.Startrek
        """
        if self._st_client_instance is None:
            from startrek_client import Startrek

            try:
                token = sdk2.Vault.data('robot-serptools', 'robot-serptools_startrek_token')
            except common.errors.VaultError as err:
                logging.error(err)
                raise common.errors.TaskFailure("Ошибка получения OAuth токена для Startrek")

            self._st_client_instance = Startrek(
                token=token,
                useragent='robot-serptools'
            )

        return self._st_client_instance

    def on_execute(self):
        self.filter_id = self.Parameters.filter_id
        result = {
            "updated": 0,
            "skipped": 0,
            "total": 0
        }

        logging.info('Fetching filter: {id}'.format(id=self.filter_id))

        st_issues = self.fetch_st_issues()
        total = len(st_issues)

        if total == 0:
            raise common.errors.TaskFailure("Не найдены задачи в фильтре")

        logging.info('Total issues: {total}'.format(total=total))
        result["total"] = total

        taskLogs = []

        for issue in st_issues:
            value = None

            # Получаем список изменений из истории по указанному полю
            for change in issue.changelog.get_all(field=self.Parameters.sourceField):
                # Запоминаем первое установленное значение
                value = self.get_history_value(change)
                if value:
                    break

            strValue = str(value) if value else 'skipped'

            if value:
                result["updated"] += 1
            else:
                result["skipped"] += 1

            taskLogs.append({"key": issue.key, "value": strValue})

        # Для режима дебага не сохраняем значения в тасках
        if self.Parameters.debugMode:
            self.Context.result_tasks = taskLogs
        else:
            self.saveTasks(taskLogs)

        self.Context.result_data = result

    def saveTasks(self, tasks):
        """Сохранение полученных значений в тасках

        :param lists tasks: Список тасок

        :return: None
        """
        for task in tasks:
            if task["value"] == 'skipped':
                continue
            issue = self._st_client.issues[task["key"]]
            fields = {}
            fields[self.Parameters.destinationField] = task["value"]
            fields['params'] = {'notify': 'false'}

            issue.update(**fields)

    def get_history_value(self, change):
        """Получение значения из записи в истории изменений таски

        :param dict change: Запись в истории таски

        :return: Значение в поле или None
        :rtype: String
        """

        if not change.fields:
            return None

        for field in change.fields:
            value = field.get('to')
            if value:
                return value

    @sdk2.header()
    def header(self):
        tasks_debug_str = ''
        if self.Context.result_tasks:
            tasks_debug_str = '<br/>'.join(
                map(
                    lambda task: '<a href="https://st.yandex-team.ru/{key}">{key}</a>: {value}'.format(
                        key=task["key"],
                        value=task["value"]
                    ),
                    self.Context.result_tasks
                )
            )

        return '<br/>'.join(["Всего задач: {total}", "Обновлено: {updated}", "Пропущено: {skipped}", tasks_debug_str]).format(
            total=self.Context.result_data["total"],
            updated=self.Context.result_data["updated"],
            skipped=self.Context.result_data["skipped"]
        ) if self.Context.result_data else ""

    def fetch_st_issues(self):
        """Загрузить данные тикетов из фильтра

        :return: Список тикетов
        :rtype: list
        """
        logging.info('Fetching filter: {id}'.format(id=self.filter_id))

        issues = self._st_client.issues.find('Filter: {id}'.format(id=self.filter_id))

        logging.info('Issues fetched: {num}'.format(num=len(issues)))

        return issues
