# -*- coding: utf-8 -*-

import json
import logging

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
import sandbox.common.types.task as ctt

from sandbox.projects.infratools.vteam.charts import base
from sandbox.projects.infratools.FetchStIssues import VteamIssuesData
import sandbox.projects.infratools.vteam.libs.issues as sti
from sandbox.projects.infratools.vteam.libs.scales import Scales, fielddate, COMMON_AGGREGATION_POINT
import sandbox.projects.infratools.vteam.libs.statface as stat
from sandbox.projects.infratools.vteam.libs.utils import russian_date


class VteamChart(sdk2.Task):
    """Базовый класс для графиков vteam. Извлекает данные из входных ресурсов и записывает выходные.
    Все расчёты делегированы конечным классам конкретных графиков.
    """

    class Parameters(sdk2.Task.Parameters):
        issues_resource_id = sdk2.parameters.Integer('Issues resource ID', required=True)

        with sdk2.parameters.RadioGroup('Scale', required=True) as scale:
            for option in Scales:
                scale.values[option.name] = scale.Value(option.value, default=option is Scales.DAILY)

        chart_id = sdk2.parameters.String('Chart ID', required=True)
        chart_title = sdk2.parameters.String('Chart title', required=False)
        chart_subtitle = sdk2.parameters.String('Chart subtitle', required=False,
                                                description="Optional explanatory information about chart")

        is_dev_mode = sdk2.parameters.Bool('Dev mode', default=False, description="If unsure, don't turn on")

    filter_id = None

    @property
    def dimensions(self):
        """Описание измерений основного отчёта в виде словаря <name: str -> type: str>"""
        return {
            'fielddate': stat.Type.DATE,
            'chart_id': stat.Type.STRING
        }

    @property
    def measures(self):
        """Описание показателей основного отчёта в виде словаря <name: str -> type: str>"""
        return {
            'filter_id': stat.Type.NUMBER,
            'chart_title': stat.Type.STRING,
            'chart_subtitle': stat.Type.STRING
        }

    @property
    def details_dimensions(self):
        """Описание измерений отчёта с расширенной информацией в виде словаря <name: str -> type: str>"""
        return {
            'fielddate': stat.Type.DATE,
            'chart_id': stat.Type.STRING,
            'issue': stat.Type.STRING
        }

    @property
    def details_measures(self):
        """Описание показателей отчёта с расширенной информацией в виде словаря <name: str -> type: str>"""
        return {
            'summary': stat.Type.STRING,
            'status': stat.Type.STRING,
            'assignee': stat.Type.STRING,
            'created': stat.Type.STRING,
            'resolved': stat.Type.STRING
        }

    def on_save(self):
        self.Requirements.semaphores = ctt.Semaphores(
            acquires=[
                ctt.Semaphores.Acquire(self.type.name, capacity=20)
            ]
        )

    def on_execute(self):
        """Достаёт данные из входного ресурса, запускает по ним расчёты и записывает результат в выходной ресурс

        :raises TaskFailure: если в указанном фильтре нет задач
        """
        resource = self.get_resource()

        self.filter_id = resource.filter_id

        issues = self.get_issues_data(resource)

        if len(issues) == 0:
            raise TaskFailure('No issues in filter %s' % self.filter_id)

        points, details = self.calculate(issues)

        self.write_data(points)
        self.write_details(details)

    def get_resource(self):
        """Ищет Sandbox ресурс с данными тикетов из Стартрека

        :rtype: sdk2.Resource
        :return: найденный ресурс

        :raises TaskFailure: если не получилось найти ресурс
        """
        resource = sdk2.Resource.find(type=VteamIssuesData, id=self.Parameters.issues_resource_id).first()

        if resource is None:
            raise TaskFailure('Resource %s with id=%s not found' % (VteamIssuesData.type,
                                                                    self.Parameters.issues_resource_id))

        logging.info('Found resource:')
        logging.info(resource)

        return resource

    def get_issues_data(self, resource):
        """Парсит переданный Sandbox ресурс и возвращает данные тикетов в виде словаря

        :param sdk2.Resource resource:

        :rtype: list
        :return: данные тикетов

        :raises TaskFailure: если не получилось распарсить данные ресурса как JSON
        """
        resource_data = sdk2.ResourceData(resource)

        try:
            issues_data = json.load(resource_data.path.open())
        except ValueError:
            raise TaskFailure('Malformed JSON in resource (id=%s)' % self.Parameters.issues_resource_id)

        return issues_data

    def calculate(self, issues):
        """Производит все необходимые вычисления для графика

        :param list issues:

        :rtype: tuple
        :return: данные графика для одного масштаба, дополнительная информация о точках

        :raises TaskFailure: в случае ошибки вычислений, должно быть выброшено в конечном методе
        """
        raise NotImplementedError('Method for chart calculations is not implemented')

    def meta(self, dimensions, measures, replace_mask=()):
        """Мета-информация о данных отчёта в формате параметров для операции Stat Upload в Нирване

        :param dict dimensions: измерения отчёта, <name:str -> type:str>
        :param dict measures: показатели отчёта, <name:str -> type:str>
        :param tuple replace_mask: дополнительные dimensions для replaceMask

        :rtype dict:
        """
        return {
            'frequencyType': self.Parameters.scale,
            'dimensionNames': dimensions.keys(),
            'dimensionTypes': dimensions.values(),
            'measuresNames': measures.keys(),
            'measuresTypes': measures.values(),
            # удаляем старые данные среза
            # отчёт за даты, которые загружались раньше, будет неактуальным, и будет конфликтовать с новыми данными
            'replaceMask': ['chart_id'] + list(replace_mask)
        }

    def point(self, date, **kwargs):
        """Конструирует данные точки основного отчёта. Подготавливает общие для всех графиков поля.

        :param datetime.datetime date: точка агрегации
        :param kwargs: специфичные для графика данные точки

        :rtype: dict
        """
        point = {
            'fielddate': fielddate(date),
            'chart_id': self.Parameters.chart_id,
            'chart_title': self.Parameters.chart_title,
            'chart_subtitle': self.Parameters.chart_subtitle,
            'filter_id': self.filter_id
        }
        point.update(kwargs)
        return point

    def details_point(self, issue, date=None, **kwargs):
        """Конструирует данные точки отчёта с дополнительной информацией. Подготавливает общие для всех графиков поля.

        :param dict issue: данные задачи
        :param datetime.datetime date: точка агрегации
        :param kwargs: специфичные для графика данные точки

        :rtype: dict
        """
        point = {
            'fielddate': fielddate(date if date else COMMON_AGGREGATION_POINT),
            'chart_id': self.Parameters.chart_id,
            'issue': sti.value(issue, 'key'),
            'summary': sti.value(issue, 'summary').replace('\n', '').replace('\t', ' ').strip(),
            'status': sti.value(issue, 'status'),
            'assignee': sti.value(issue, 'assignee'),
            'created': russian_date(sti.value(issue, 'createdAt')),
            'resolved': russian_date(sti.value(issue, 'resolvedAt'))
        }
        point.update(kwargs)
        return point

    def write_data(self, points):
        """Создаёт ресурсы для основного отчёта

        :param list points: данные графика
        """
        self.write_resource(base.VteamChartData, 'Chart JSON', 'chart_data.json', points)
        self.write_resource(base.VteamChartMeta, 'Chart JSON Meta', 'chart_meta.json',
                            self.meta(self.dimensions, self.measures))

    def write_details(self, details=None):
        """Создаёт ресурсы для отчёта с дополнительной информацией, если она есть

        :param list details: дополнительная информация для графика
        """
        if details is None:
            return

        self.write_resource(base.VteamChartDetails, 'Chart Details JSON', 'chart_details_data.json', details)
        # При аплоаде более 10к строк через кубик хитмана данные льются чанками. А replace_mask просто прокидывается
        # между заливками. И если в нём нет fielddate, то каждый следующий чанк очистит предыдущие.
        # Поскольку в доп.отчёте одна строка на одну задачу для каждой даты, строк может быть больше 10к.
        self.write_resource(base.VteamChartDetailsMeta, 'Chart Details JSON Meta', 'chart_details_meta.json',
                            self.meta(self.details_dimensions, self.details_measures, replace_mask=('fielddate',)))

    def write_resource(self, resource_cls, description, path, data):
        """Создаёт ресурс переданного типа с данными

        :param type resource_cls: класс создаваемого ресурса
        :param str description: описание
        :param str path: путь к файлу ресурса
        :param data: данные для записи
        """
        data_json = json.dumps(data)

        resource = resource_cls(self, description, path,
                                filter_id=self.filter_id,
                                chart_type=self.type.name,
                                scale=self.Parameters.scale)

        resource_data = sdk2.ResourceData(resource)
        resource_data.path.write_bytes(data_json)
