# -*- coding: utf-8 -*-

import logging
import textwrap

from sandbox import sdk2
from sandbox.common.errors import TaskFailure

# если импортировать сам класс, он станет отдельным таском в SB
from sandbox.projects.infratools.vteam.charts import base
from sandbox.projects.infratools.vteam.libs.scales import get_aggregation_point
import sandbox.projects.infratools.vteam.libs.issues as sti
import sandbox.projects.infratools.vteam.libs.statface as stat
from sandbox.projects.infratools.vteam.libs.worktime import worktime
from sandbox.projects.infratools.vteam.libs.series import from_json


class VteamCycletimeChart(base.VteamChart):
    """VTeam - Циклтайм"""

    @property
    def dimensions(self):
        dimensions = super(VteamCycletimeChart, self).dimensions
        dimensions['slice'] = stat.Type.STRING
        dimensions['percentile'] = stat.Type.NUMBER
        dimensions['require_pr'] = stat.Type.STRING
        return dimensions

    @property
    def measures(self):
        return {
            'seconds': stat.Type.NUMBER,
            'issues': stat.Type.NUMBER
        }

    class Parameters(base.VteamChart.Parameters):
        series_json = sdk2.parameters.String('Series', multiline=True, required=True, description='JSON')

    class Context(sdk2.Task.Context):
        report = {}

    def calculate(self, issues):
        try:
            series_parsed = from_json(self.Parameters.series_json)
        except ValueError:
            raise TaskFailure('Malformed series JSON')

        issues_pr_filter = {
            stat.Boolean.true: filter(sti.has_pr, issues),
            stat.Boolean.false: issues
        }

        points = [
            self.point(
                date,
                slice=series.title,
                percentile=percentile.threshold,
                require_pr=require_pr,
                seconds=percentile.calc(values),
                issues=len(values)
            )
            for series in series_parsed
            for require_pr, issues_filtered in issues_pr_filter.iteritems()
            for date, values in self.calc_slice(issues_filtered, series).iteritems()
            for percentile in series.fn
        ]

        self.log_context()

        return points, None

    def calc_slice(self, issues, series):
        """Вычисляет срез циклтайма.

        Возвращает словарь, в котором ключ — это точка на графике (дата), а значение — список значений циклтайма
        для задач, которые попали в эту точку.

        Точка на графике определяется концом среза.

        Значение циклтайма для задачи вычисяется как время между началом среза и концом среза, исключая выходные.

        :param list issues: задачи, по которым нужно посчитать циклтайм
        :param sandbox.projects.infratools.vteam.libs.series.IntervalSeries series: параметры среза

        :rtype: dict
        """
        points = {}
        skipped = []
        failed = []

        for issue in issues:
            start, end = series.boundaries(issue)

            if not start or not end:
                skipped.append(sti.value(issue, 'key'))
                continue

            if start > end:
                failed.append(sti.value(issue, 'key'))
                continue

            key = get_aggregation_point(self.Parameters.scale, end)

            if key not in points:
                points[key] = []

            points[key].append(worktime(start, end).total_seconds())

        self.update_context(series.title, len(issues), skipped, failed)

        return points

    def update_context(self, series_title, total, skipped=None, failed=None):
        """Дополняет контекст статистикой о пропущенных и некорректных задачах в обработанном срезе.

        :param str series_title: название среза
        :param int total: общее кол-во задач в срезе
        :param list skipped: список ключей пропущенных задач
        :param list failed: список ключей задач с отрицательным циклтаймом
        """
        if series_title not in self.Context.report:
            self.Context.report[series_title] = {
                'total': 0,
                'skipped': [],
                'failed': []
            }

        series_report = self.Context.report[series_title]

        # расчёт идёт по двум дименшенам: с учётом прилинкованного к задаче PR и без
        # отчёт пишется для дименшена "без учёта PR", но контекст обновляется в обоих случаях
        # если количество задач больше, то это нефильтрованный список, который нужно отразить в отчёте
        if total > series_report['total']:
            series_report['total'] = total

        if skipped:
            series_report['skipped'] = list(set(series_report['skipped'] + skipped))

        if failed:
            series_report['failed'] = list(set(series_report['failed'] + failed))

    def log_context(self):
        """Записывает в лог финальную версию собранной статистики о пропущенных и некорректных задачах"""
        for series_title, series_report in self.Context.report.iteritems():
            logging.info('=== %s' % series_title)
            logging.info('    Total issues: %s' % series_report['total'])

            skipped = series_report['skipped']
            if skipped:
                logging.info('    === Skipped: %s' % len(skipped))
                logging.info(', '.join(skipped))

            failed = series_report['failed']
            if failed:
                logging.info('    === With negative cycle time: %s' % len(failed))
                logging.info(', '.join(failed))

    @sdk2.header()
    def header(self):
        if not self.Context.report:
            return None

        header = ''

        for series_title, series_report in self.Context.report.iteritems():
            header += textwrap.dedent("""
                <b>{series}</b><br>
                Skipped issues: {skipped}/{total}<br>
                Negative cycle time: {failed}<br>
                {failed_keys}<br>
                <br>
            """).format(
                series=series_title.encode('utf-8'),
                total=series_report['total'],
                skipped=len(series_report['skipped']),
                failed=len(series_report['failed']),
                failed_keys=', '.join(series_report['failed'])
            )

        header += '<br>See common log for more info'

        return header
