# -*- coding: utf-8 -*-

from __future__ import division
from enum import Enum
from sandbox import sdk2

# если импортировать сам класс, он станет отдельным таском в SB
from sandbox.projects.infratools.vteam.charts import base
from sandbox.projects.infratools.vteam.libs.scales import get_keys, get_start_point
import sandbox.projects.infratools.vteam.libs.issues as sti
import sandbox.projects.infratools.vteam.libs.statface as stat


class Fields(Enum):
    priority = 'Priority'
    issue_weight = 'Issue Weight'

    __order__ = 'priority issue_weight'


class VteamMedianKgbChart(base.VteamChart):
    """VTeam - Median KGB"""

    @property
    def dimensions(self):
        dimensions = super(VteamMedianKgbChart, self).dimensions
        dimensions['source'] = stat.Type.STRING
        dimensions['history'] = stat.Type.STRING
        return dimensions

    measures = {
        'team_weight': stat.Type.NUMBER,
        'team_total_weight': stat.Type.NUMBER,
        'team_issues': stat.Type.STRING,
        'team_priorities': stat.Type.STRING,
        'other_weight': stat.Type.NUMBER,
        'other_total_weight': stat.Type.NUMBER,
        'other_issues': stat.Type.STRING,
        'other_priorities': stat.Type.STRING,
        'total_weight': stat.Type.NUMBER,
    }

    class Parameters(base.VteamChart.Parameters):
        with sdk2.parameters.RadioGroup('Field', required=True) as field:
            for option in Fields:
                field.values[option.name] = field.Value(option.value, default=option is Fields.priority)

        with_history = sdk2.parameters.Bool('With history', default=True, required=False)

    def calculate(self, issues):
        details = [self.details_point(issue) for issue in issues]
        start_date = get_start_point(self.Parameters.scale, sti.dates(issues, 'resolvedAt'))

        if start_date is None:
            start_date = min(sti.dates(issues, 'createdAt'))

        dates = get_keys(self.Parameters.scale, start=start_date)
        groupped_issues = self.group_issues_by_bug_source(issues)
        points = self.calculate_points(
            dates,
            groupped_issues.get('team'),
            groupped_issues.get('other')
        )

        return points, details

    def calculate_points(self, dates, team_issues, other_issues):
        points = []
        is_value_started = False

        team_distributed_issues_weight = 0
        other_distributed_issues_weight = 0
        distributed_index = 0

        for i in range(1, len(dates)):
            current_date = dates[i - 1]

            team_issues_data = self.prepare_issues_by_date(dates[i], team_issues)
            other_issues_data = self.prepare_issues_by_date(dates[i], other_issues)

            team_total_weight = team_issues_data.get('total_weight')
            other_total_weight = other_issues_data.get('total_weight')

            # Обрезаем нулевые значения в начале графика.
            # График строится от первой даты открытия задачи. Задача может быть создана давно, а взята в работу недавно.
            # В этом случае в начале графика будут нули. В идеале нужно брать от даты первого проставления поля weight,
            # но это усложнение с точки зрения кода.
            if team_total_weight == 0 and other_total_weight == 0 and not is_value_started:
                continue

            is_value_started = True

            # Расчитываем коэффициенты для сглаживания весов
            # k_n^−1 = 1 + 0.9 + 0.9×0.9 + ... + 0.9^(n−1)
            team_distributed_issues_weight = team_distributed_issues_weight * 0.9 + team_total_weight
            other_distributed_issues_weight = other_distributed_issues_weight * 0.9 + other_total_weight
            distributed_index = distributed_index * 0.9 + 1

            team_rounded_issues_weight = int(round(team_distributed_issues_weight / distributed_index))
            other_rounded_issues_weight = int(round(other_distributed_issues_weight / distributed_index))

            points.append(self.point(
                current_date,

                source=Fields[self.Parameters.field].value,
                history=stat.Boolean.true if self.Parameters.with_history else stat.Boolean.false,

                total_weight=team_rounded_issues_weight + other_rounded_issues_weight,

                team_weight=team_rounded_issues_weight,
                team_total_weight=team_total_weight,
                team_issues=','.join(team_issues_data.get('issues')),
                team_priorities=','.join(team_issues_data.get('priorities')),

                other_weight=other_rounded_issues_weight,
                other_total_weight=other_total_weight,
                other_issues=','.join(other_issues_data.get('issues')),
                other_priorities=','.join(other_issues_data.get('priorities')),
            ))

        return points

    def prepare_issues_by_date(self, next_date, issues):
        data = {
            'total_weight': 0,
            'priorities': [],
            'issues': [],
        }

        for issue in issues:
            created = sti.value(issue, 'createdAt')
            closed = sti.value(issue, 'resolvedAt') or next_date

            # учитываем только задачи, которые были открыты в конце текущего интервала
            if created >= next_date or closed < next_date:
                continue

            data['total_weight'] += self.get_weight_by_parameter(issue, next_date)
            data['priorities'].append(sti.value(issue, 'priority'))
            data['issues'].append(sti.value(issue, 'key'))

        return data

    def group_issues_by_bug_source(self, issues):
        groupped_issues = {'team': [], 'other': []}

        for issue in issues:
            is_found_by_team = int(u'Команда' in (sti.value(issue, 'bugSource') or []))

            if is_found_by_team:
                groupped_issues['team'].append(issue)
            else:
                groupped_issues['other'].append(issue)

        return groupped_issues

    def get_weight_by_parameter(self, issue, date):
        """Вычисляет значимый вес в зависимости от указанного параметра

        :param dict issue:
        :param datetime.datetime date:

        :rtype: int
        :return: вес задачи по нужному полю
        """
        if Fields[self.Parameters.field] is Fields.priority:
            return self.get_priority(issue, date)
        else:
            return self.get_weight(issue, date)

    def get_weight(self, issue, date):
        """Вычисляет значение issueWeight
        В зависимости от параметра учитывает историю или нет

        :param dict issue:
        :param datetime.datetime date:

        :rtype: int
        :return: вес задачи
        """
        if self.Parameters.with_history:
            weight = sti.last_value(issue, 'issueWeight', before=date)
        else:
            weight = sti.value(issue, 'issueWeight')

        return weight or 0

    def get_priority(self, issue, date):
        """Вычисляет значение priority
        В зависимости от параметра учитывает историю или нет

        :param dict issue:
        :param datetime.datetime date:

        :rtype: int
        :return: вес задачи по полю priority
        """
        if self.Parameters.with_history:
            priority = sti.last_value(issue, 'priority', before=date)
        else:
            priority = sti.value(issue, 'priority')

        return sti.weight_by_priority(priority) or 0
