# -*- coding: utf-8 -*-

from datetime import datetime

from sandbox import sdk2
from sandbox.common.errors import TaskFailure

# если импортировать сам класс, он станет отдельным таском в SB
from sandbox.projects.infratools.vteam.charts import base_new
import sandbox.projects.infratools.vteam.libs.issues as sti
import sandbox.projects.infratools.vteam.libs.statface as stat
from sandbox.projects.infratools.vteam.libs.scales import get_keys, get_start_point, MOSCOW_TZ
from sandbox.projects.infratools.vteam.libs.worktime import worktime, bound
from sandbox.projects.infratools.vteam.libs.series import from_json
from sandbox.projects.infratools.vteam.libs.utils import russian_date


class VteamSlaChart(base_new.VteamChart):
    """VTeam - SLA"""

    @property
    def dimensions(self):
        base = super(VteamSlaChart, self).dimensions
        base.update({
            'series_id': stat.Type.STRING,
            'fn': stat.Type.STRING,
            'exclude_holidays': stat.Type.STRING
        })
        return base

    @property
    def measures(self):
        base = super(VteamSlaChart, self).measures
        base.update({
            'series_name': stat.Type.STRING,
            'series_pos': stat.Type.NUMBER,
            'seconds': stat.Type.NUMBER,
            'goal': stat.Type.NUMBER
        })
        return base

    @property
    def details_dimensions(self):
        base = super(VteamSlaChart, self).details_dimensions
        base.update({
            'series_id': stat.Type.STRING,
            'exclude_holidays': stat.Type.STRING
        })
        return base

    @property
    def details_measures(self):
        base = super(VteamSlaChart, self).details_measures
        base.update({
            'sla_start': stat.Type.STRING,
            'sla_end': stat.Type.STRING,
            'duration': stat.Type.NUMBER,
            'time_spent': stat.Type.NUMBER  # артефакт от SLA с time spent, нужно избавиться (?)
        })
        return base

    class Parameters(base_new.VteamChart.Parameters):
        series_json = sdk2.parameters.String('Series', multiline=True, required=True, description='JSON')

    def calculate(self, issues):
        points = []
        details = []

        start_date = get_start_point(self.Parameters.scale, sti.dates(issues, 'createdAt'))
        dates = get_keys(self.Parameters.scale, start=start_date)

        try:
            series_parsed = from_json(self.Parameters.series_json)
        except ValueError:
            raise TaskFailure('Malformed series JSON')

        for i in range(1, len(dates)):
            current_date = dates[i - 1]
            next_date = dates[i]

            for series in series_parsed:
                for exclude_holidays in [True, False]:  # считаем сразу оба варианта
                    durations = []

                    for issue in issues:
                        if not series.includes(issue, next_date):
                            continue

                        sla_start, sla_end = series.boundaries(issue)

                        if not sla_start:
                            continue

                        if sla_start >= next_date:
                            continue

                        current_start, current_end = bound(sla_start, sla_end or datetime.now(MOSCOW_TZ),
                                                           upper=next_date)
                        current_duration = worktime(current_start, current_end, exclude_holidays).total_seconds()
                        # артефакт от SLA с time spent, нужно избавиться (?)
                        current_time_spent = sti.last_value(issue, 'spent', before=next_date)

                        durations.append(current_duration)

                        details += [self.details_point(
                            issue,
                            current_date,
                            series_id='%s_%s' % (self.Parameters.chart_id, series.id),
                            exclude_holidays=stat.Boolean.true if exclude_holidays else stat.Boolean.false,
                            sla_start=russian_date(sla_start),
                            sla_end=russian_date(sla_end),
                            duration=current_duration,
                            # артефакт от SLA с time spent, нужно избавиться (?)
                            time_spent=current_time_spent.total_seconds() if current_time_spent else None
                        )]

                    if durations:
                        points += [self.point(
                            current_date,
                            series_id='%s_%s' % (self.Parameters.chart_id, series.id),
                            series_name=series.title,
                            series_pos=series.position,
                            fn=fn.title,
                            exclude_holidays=stat.Boolean.true if exclude_holidays else stat.Boolean.false,
                            seconds=fn.calc(durations),
                            goal=series.goal
                        ) for fn in series.fn]

        return points, details
