# -*- coding: utf-8 -*-

from sandbox import sdk2
from sandbox.common.errors import TaskFailure

# если импортировать сам класс, он станет отдельным таском в SB
from sandbox.projects.infratools.vteam.charts import base_new
import sandbox.projects.infratools.vteam.libs.issues as sti
import sandbox.projects.infratools.vteam.libs.statface as stat
from sandbox.projects.infratools.vteam.libs.scales import get_keys, get_start_point
from sandbox.projects.infratools.vteam.libs.series import from_json, SlaSeries
from sandbox.projects.infratools.vteam.libs.utils import russian_date


class ToolsSlaChart(base_new.VteamChart):
    """Tools SLA"""

    @property
    def dimensions(self):
        base = super(ToolsSlaChart, self).dimensions
        base.update({
            'series_id': stat.Type.STRING,
            'fn': stat.Type.STRING
        })
        return base

    @property
    def measures(self):
        base = super(ToolsSlaChart, self).measures
        base.update({
            'series_name': stat.Type.STRING,
            'series_pos': stat.Type.NUMBER,
            'ms': stat.Type.NUMBER,
            'goal_ms': stat.Type.NUMBER
        })
        return base

    @property
    def details_dimensions(self):
        base = super(ToolsSlaChart, self).details_dimensions
        base.update({
            'series_id': stat.Type.STRING,
            'exclude_holidays': stat.Type.STRING
        })
        return base

    @property
    def details_measures(self):
        base = super(ToolsSlaChart, self).details_measures
        base.update({
            'sla_start': stat.Type.STRING,
            'sla_end': stat.Type.STRING,
            'duration': stat.Type.NUMBER
        })
        return base

    class Parameters(base_new.VteamChart.Parameters):
        series_json = sdk2.parameters.String('Series', multiline=True, required=True, description='JSON')

    def calculate(self, issues):
        points = []
        details = []

        start_date = get_start_point(self.Parameters.scale, sti.dates(issues, 'createdAt'))
        dates = get_keys(self.Parameters.scale, start=start_date)

        try:
            series_parsed = from_json(self.Parameters.series_json)
        except ValueError:
            raise TaskFailure('Malformed series JSON')

        for i in range(1, len(dates)):
            current_date = dates[i - 1]
            next_date = dates[i]

            for series in series_parsed:
                if not isinstance(series, SlaSeries):
                    raise TaskFailure('Wrong series type, expected ValueSeries, got %s' % type(series))

                durations = []
                sla_goal = None

                for issue in issues:
                    try:
                        is_issue_in_series = series.includes(issue, next_date)
                    except ValueError:
                        # no target sla tracker on current issue, skipping
                        continue

                    if not is_issue_in_series:
                        continue

                    current_duration = series.value(issue)
                    sla_start, sla_end = series.boundaries(issue)

                    if sla_goal is None:
                        sla_goal = series.sla_goal(issue)

                    durations.append(current_duration)

                    details += [self.details_point(
                        issue,
                        current_date,
                        series_id='%s_%s' % (self.Parameters.chart_id, series.id),
                        sla_start=russian_date(sla_start),
                        sla_end=russian_date(sla_end),
                        duration=current_duration
                    )]

                if durations:
                    points += [self.point(
                        current_date,
                        series_id='%s_%s' % (self.Parameters.chart_id, series.id),
                        series_name=series.title,
                        series_pos=series.position,
                        fn=fn.title,
                        ms=fn.calc(durations),
                        goal_ms=None if series.goal is False else sla_goal
                    ) for fn in series.fn]

        return points, details
