# -*- coding: utf-8 -*-

import json
import numpy as np
from datetime import datetime, timedelta

# если импортировать сам класс, он станет отдельным таском в SB
from sandbox.projects.infratools.vteam.charts import base
from sandbox.projects.infratools.vteam.libs.scales import get_keys, get_start_point
import sandbox.projects.infratools.vteam.libs.issues as sti
import sandbox.projects.infratools.vteam.libs.statface as stat
from sandbox.projects.infratools.vteam.libs.worktime import bound, MOSCOW_TZ


class VteamWipChart(base.VteamChart):
    """VTeam - WIP"""

    measures = {
        'efficiency': stat.Type.NUMBER,
        'efficiency_with_backlog': stat.Type.NUMBER
    }

    @property
    def details_measures(self):
        measures = super(VteamWipChart, self).details_measures
        measures['efficiency'] = stat.Type.NUMBER
        measures['statuses'] = stat.Type.STRING
        measures['efficiency_with_backlog'] = stat.Type.NUMBER
        measures['statuses_with_backlog'] = stat.Type.STRING
        return measures

    def calculate(self, issues):
        points = []
        details = []
        start_date = get_start_point(self.Parameters.scale, sti.dates(issues, 'createdAt'))
        dates = get_keys(self.Parameters.scale, start=start_date)

        for i in range(1, len(dates)):
            current_date = dates[i - 1]
            next_date = dates[i]
            current_data = []

            for issue in issues:
                if not sti.was_open_between(issue, current_date, next_date):
                    continue

                issue_data = self.details_point(issue, current_date)

                backlog_statuses = sti.stage_statuses(issue, 'backlog')
                start_progress = sti.progress_start(issue, backlog_statuses)
                start_backlog = sti.value(issue, 'createdAt')
                end = sti.progress_end(issue) or datetime.now(MOSCOW_TZ)

                # если нет даты старта прогресса, то задача либо не выходила из бэклога, либо сразу была закрыта
                # такие задачи никак не нужно учитывать на линии без бэклога, искусственный 0 или 1 для таких задач
                # сделают расчёт некорректным
                if start_progress:
                    start_progress, end = bound(start_progress, end, upper=next_date)
                    progress, total, statuses = self.time(issue, start_progress, end, exclude=backlog_statuses)
                    issue_data['efficiency'] = progress / total if total else 0
                    issue_data['statuses'] = json.dumps(statuses)

                start_backlog, end = bound(start_backlog, end, upper=next_date)
                progress, total, statuses = self.time(issue, start_backlog, end)
                issue_data['efficiency_with_backlog'] = progress / total if total else 0
                issue_data['statuses_with_backlog'] = json.dumps(statuses)

                current_data.append(issue_data)

            if not current_data:
                continue

            details += current_data

            efficiencies = [item.get('efficiency') for item in current_data if 'efficiency' in item]
            efficiencies_with_backlog = [item.get('efficiency_with_backlog', 0) for item in current_data]
            points.append(self.point(
                current_date,
                efficiency=np.percentile(efficiencies, 75, interpolation='higher') if efficiencies else None,
                efficiency_with_backlog=np.percentile(efficiencies_with_backlog, 75,
                                                      interpolation='higher') if efficiencies_with_backlog else None
            ))

        return points, details

    @staticmethod
    def time(issue, start, end, exclude=()):
        """Считает время задачи в рабочих статусах и общее время жизни задачи. Собирает время в каждом из статусов,
        помечая его как время работы или время ожидания. Если задан exclude, указанные там статусы проигнорируются.

        :param dict issue:
        :param datetime start:
        :param datetime end:
        :param tuple exclude: статусы, которые не нужно учитывать

        :rtype: tuple
        :return: секунды в рабочих статусах, секунды жизни задачи, распределение времени по статусом с меткой стадии
        """
        progress_time = timedelta()
        total_time = timedelta()
        statuses = {}

        time_in_statuses = sti.time_in_statuses(issue, start, end)
        progress_statuses = sti.stage_statuses(issue, 'progress')

        for status, duration in time_in_statuses.iteritems():
            if status in exclude:
                continue

            # Если время в статусе меньше минуты, это может быть один из случаев:
            # 1. Погрешность между выставлением статуса с резолюцией и присвоения даты резолюции. Нужно скипнуть.
            # 2. Случайно перевели в неправильный статус и поменяли на правильный. Можно скипнуть.
            # Минута взята с потолка.
            if duration < timedelta(minutes=1):
                continue

            status_type = 'wait'

            if status in progress_statuses:
                progress_time += duration
                status_type = 'work'

            statuses[status] = {
                'duration': duration.total_seconds(),
                'type': status_type
            }

            total_time += duration

        return progress_time.total_seconds(), total_time.total_seconds(), statuses
