# -*- coding: utf-8 -*-

from datetime import datetime, timedelta

from .utils import parse_date, leading_num, parse_iso_duration
from .worktime import MOSCOW_TZ, bound, worktime

# see https://st.yandex-team.ru/FEI-10379
STAGE_STATUSES = {
    'SIDEBYSIDE': {
        'progress': ('experimentOn', 'readyForCheck'),
        'closed': ('experimentOff', 'closed')
    },
    'EXPERIMENTS': {
        'progress': ('launched', 'published', 'completed', 'inProgress'),
        'closed': ('readyForRelease', 'closed')
    },
    'SERPDESIGN': {
        'progress': ('inProgress', 'inReview'),
        'closed': ('readyForDevelopment', 'inDevelopment', 'closed')
    },
    'SERP': {
        'progress': ('inProgress', 'inReview', 'testing'),
        'closed': ('readyForDev', 'dev', 'readyForRc', 'rc', 'released', 'closed')
    },
    '_DEFAULT': {
        'backlog': ('new', 'open'),
        'progress': ('inProgress',),
        'closed': ('closed',)
    }
}


def _parse_field(field, field_id):
    """Извлекает значение поля из структуры, описывающей поле в данных задачи.
    У разных полей разная структура. Например, для статуса это словарь с полями id, key и display,
    из которых нам нужен key.

    :param field:
    :param str field_id:

    :return: значение поля, приведённое к формату, в котором оно используется в расчётах
    """
    if field_id in ('createdAt', 'resolvedAt', 'updatedAt', 'start', 'end'):
        return parse_date(field)
    elif field_id in ('issueWeight',):
        return leading_num(field)
    elif field_id in ('queue', 'priority', 'status', 'resolution'):
        return (field or {}).get('key')
    elif field_id in ('assignee',):
        return (field or {}).get('id')
    elif field_id in ('key', 'summary', 'bugSource', 'changelog', 'remotelinks', 'causeOfReleaseBug', 'autotesting', 'tags', 'sla'):
        return field
    elif field_id in ('spent',):
        return parse_iso_duration(field)
    elif field_id in ('components',):
        return map(lambda val: val.get('id'), field) if field else field
    else:
        raise NotImplementedError('Parser is not implemented for field "%s"' % field_id)


def dates(issues, field_id):
    """Извлекает из данных тикетов даты из указанного поля и формирует список, пропуская тикеты, у которых
    указанное поле пустое.

    :param list issues: список тикетов
    :param str field_id: поле, которое надо распарсить как дату

    :rtype: list
    """
    return [date for date in (value(issue, field_id) for issue in issues) if date]


def value(issue, field_id):
    """Текущее значение поля задачи

    :param dict issue: данные задачи
    :param str field_id: ключ поля

    :return: значение поля
    """
    return _parse_field(issue.get(field_id), field_id)


def last_value(issue, field_id, before=datetime.now(MOSCOW_TZ)):
    """Ищёт последнее значение, которое было указано в поле до момента `before`.

    :param dict issue: данные задачи
    :param str field_id: ключ поля
    :param datetime before: временная метка, верхняя граница для поиска изменений поля

    :return: значение или None
    """
    if value(issue, 'createdAt') >= before:
        return None

    field_clog = flat_changelog(issue, field_id)
    clog_before_time = filter(lambda item: item.get('date') < before, field_clog)

    if clog_before_time:
        return clog_before_time[-1].get('to')
    else:
        return initial_value(issue, field_id, changelog=field_clog)


clog_cache = {}


def flat_changelog(issue, field_id=None):
    """Строит плоский список изменений полей задачи, с опциональным фильтром по полю.
    Поля from и to элементов списка содержат распаршенное значение полей.

    :param dict issue: данные задачи
    :param str field_id: id поля, по которому нужно отфильтровать ченжлог

    :rtype: list
    """
    key = value(issue, 'key')

    if key not in clog_cache:
        clog_cache[key] = {}

    if field_id not in clog_cache[key]:
        clog_cache[key][field_id] = sorted([{
            'date': value(change, 'updatedAt'),
            'field': field.get('field').get('id'),
            'from': _parse_field(field.get('from'), field.get('field').get('id')),
            'to': _parse_field(field.get('to'), field.get('field').get('id'))
        }
            for change in value(issue, 'changelog')
            for field in change.get('fields')
            if field_id is None or field.get('field').get('id') == field_id
        ], key=lambda x: x.get('date'))

    return clog_cache[key][field_id]


def initial_value(issue, field_id, changelog=None):
    """Возвращает значение поля, которое было у задачи про создании.

    :param dict issue: объект задачи
    :param str field_id: поле, значение которого надо найти
    :param list changelog: ченжлог по нужному полю, если он уже был вычислен, чтобы не собирать его повторно

    :rtype object:
    :return: объект значения или None, если поле не было заполнено при создании задачи
    """
    if changelog is None:
        changelog = flat_changelog(issue, field_id)
    return changelog[0].get('from') if changelog else value(issue, field_id)


def _matches_or_not_empty(val, target_values=(), fn=None):
    """Проверяет, удовлетворяет ли значение поля условиям поиска в истории.
    Если есть список искомых значений, значение должно быть в нём. Если список пуст, подойдёт любое непустое значение.
    Если передана функция, то список значений игнорируется, всегда возвращается результат этой функции.

    :param val: значение поля
    :param tuple target_values: искомые значения
    :param function fn: булевая функция для проверки значения по проивзольной логике

    :rtype: bool
    """
    if fn:
        return fn(val)

    if target_values:
        return val in target_values

    return val is not None and val != ''


def first_change(issue, field_id, to=(), fn=None):
    """Находит дату первого изменения поля.

    :param dict issue: данные задачи
    :param str field_id: ключ поля
    :param tuple to: список значений, перевод в которые надо искать. Если пустой, подходит любое непустое значение.
    :param function fn: булевая функция для проверки значения по проивзольной логике, приоритетнее списка значений

    :rtype: datetime
    :return: дата первого изменения поля или None
    """
    changelog = flat_changelog(issue, field_id)

    changelog.insert(0, {
        'to': initial_value(issue, field_id, changelog=changelog),
        'date': value(issue, 'createdAt')
    })

    for change in changelog:
        if _matches_or_not_empty(change.get('to'), to, fn):
            return change.get('date')


def _get_changelog_by_prop(change, field_id, prop):
    """
    :param dict change:
    :param str field_id:
    :param str prop:

    :rtype: str
    """
    # У таких свойств как компоненты в "to" содержится список значений проставленных на момент ченджлога.
    # Необходимо выяснить, какие компоненты добавились или удалились.
    if field_id == 'components':
        to_values = change.get('to') or []
        from_values = change.get('from') or []

        added = list(set(to_values).difference(from_values))
        removed = list(set(from_values).difference(to_values))

        return added + removed

    return change.get(prop)


def last_change(issue, field_id, to=(), fn=None, stack_changes=True, require_current=False):
    """Находит дату последнего изменения поля.

    :param dict issue: данные задачи
    :param str field_id: ключ поля
    :param tuple to: список значений, перевод в которые надо искать. Если пустой, подходит любое непустое значение.
    :param function fn: булевая функция для проверки значения по проивзольной логике, приоритетнее списка значений
    :param bool stack_changes: если True, то при изменение одинаковых полей, например резолюции, будет использоваться дата первого изменения.
    :param bool require_current: если True, текущее значение поля должно быть среди искомых

    :rtype: datetime
    :return: дата последнего изменения поля или None
    """
    changelog = flat_changelog(issue, field_id)

    changelog.insert(0, {
        'to': initial_value(issue, field_id, changelog=changelog),
        'date': value(issue, 'createdAt')
    })

    last_change_val = _get_changelog_by_prop(changelog[-1], field_id, 'to')
    if require_current and not _matches_or_not_empty(last_change_val, to, fn):
        return None

    if not stack_changes:
        for change in changelog[::-1]:
            last_change_val = _get_changelog_by_prop(change, field_id, 'to')
            if _matches_or_not_empty(last_change_val, to, fn):
                return change.get('date')

        return None

    last_date = None
    is_stacked = False

    for change in changelog:
        last_change_val = _get_changelog_by_prop(change, field_id, 'to')
        if not _matches_or_not_empty(last_change_val, to, fn):
            is_stacked = False
        elif not is_stacked:
            last_date = change.get('date')
            is_stacked = True

    return last_date


def has_pr(issue):
    """Определяет, прилинкован ли к задаче пул реквест.

    :param dict issue:

    :rtype: bool
    """
    # TODO нужно найти способ узнавать, что задача связана с PR на bitbucket
    # он не кладётся в remotelinks, его вообще нет в данных задачи на момент написания
    # см. https://st.yandex-team.ru/STARTREKAPI-2
    return bool(value(issue, 'remotelinks'))


def was_open_between(issue, start, end):
    """Определяет, была ли задача открыта в заданный временной интервал.

    :param dict issue:
    :param datetime start:
    :param datetime end:

    :rtype: bool
    """
    created_at = value(issue, 'createdAt')
    resolved_at = value(issue, 'resolvedAt') or datetime.now(MOSCOW_TZ)
    return created_at < end and resolved_at >= start


def progress_start(issue, backlog_statuses=None, closed_statuses=('closed',)):
    """Возвращает дату начала работы над задачей. Определяется переводом задачи из бэклога; обычно - в любой статус,
    кроме New или Open. Также по умолчанию игнорируется статус Closed, чтобы в задачах, которые были закрыты
    без старта работ (например, с Won't fix), даты начала работ не было.

    :param dict issue: данные задачи
    :param tuple backlog_statuses: статусы, означающие нахождение задачи в бэклоге
    :param tuple closed_statuses: статусы, означающие окончание работы над задачей

    :rtype: datetime or None
    """
    if backlog_statuses is None:
        backlog_statuses = stage_statuses(issue, 'backlog')

    changelog = flat_changelog(issue, 'status')

    changelog.insert(0, {
        'to': initial_value(issue, 'status', changelog=changelog),
        'date': value(issue, 'createdAt')
    })

    excluded = backlog_statuses + closed_statuses

    for change in changelog:
        if change['to'] not in excluded:
            return change['date']


def progress_end(issue):
    """Возвращает дату окончания общего цикла работы над задачей. Стандартно, это дата resolvedAt.

    :param dict issue:

    :rtype: datetime or None
    """
    return value(issue, 'resolvedAt')


def stage_statuses(issue, stage):
    """Возващает набор статусов для очереди, определяющий стадию работы над задачей.
    Если нет конфига для очереди вообще, или если нет конфига для стадии у нужной очереди,
    возвращается набор статусов для этой стадии по умолчанию.

    :param dict issue:
    :param str stage: тип статусов

    :rtype: tuple

    :raises KeyError: когда передана некорректная стадия
    """
    default_statuses = STAGE_STATUSES['_DEFAULT']
    queue_statuses = STAGE_STATUSES.get(issue.get('queue').get('key'), default_statuses)

    try:
        return queue_statuses.get(stage, default_statuses[stage])
    except KeyError:
        raise TypeError('Unknown stage "%s" (one of "%s" expected)' % (stage, ', '.join(queue_statuses.keys())))


def time_in_statuses(issue, lower=None, upper=None):
    """Собирает для задачи время нахождения в каждом статусе.

    :param dict issue:
    :param datetime lower: ниже этой даты время в статусах не учитывается
    :param datetime upper: выше этой даты время в статусах не учитывается

    :rtype: dict
    :returns: словарь <ключ статуса => timedelta>
    """
    statuses = {}
    changelog = flat_changelog(issue, 'status')

    # последняя (или единственная) запись — для учёта времени в последнем/текущем статусе
    changelog.append({
        'date': datetime.now(MOSCOW_TZ),
        'from': changelog[-1].get('to') if changelog else value(issue, 'status')
    })

    # первая запись с датой создания — для консистентного источника стартовой даты в цикле
    changelog.insert(0, {
        'date': value(issue, 'createdAt')
    })

    for i in range(1, len(changelog)):
        change = changelog[i]
        prev_change = changelog[i - 1]

        status = change.get('from')

        if status not in statuses:
            statuses[status] = timedelta()

        start, end = bound(prev_change.get('date'), change.get('date'), lower, upper)

        statuses[status] += worktime(start, end)

    return statuses


def weight_by_priority(priority):
    """Ассоциация значения поля Priority задачи с весом.

    :param str priority: ключ приоритета

    :rtype: int
    """
    return {
        'blocker': 100,
        'critical': 100,
        'normal': 10,
        'minor': 1,
        'trivial': 1
    }.get(priority)
