# -*- coding: utf-8 -*-

from enum import Enum
from datetime import datetime
from dateutil.relativedelta import relativedelta, MO

from .worktime import MOSCOW_TZ


# ключи должны совпадать с возможными значениями поля frequencyType операции Stat Upload
# https://wiki.yandex-team.ru/hitman/nirvana/stat/
class Scales(Enum):
    DAILY = 'DAILY'
    WEEKLY = 'WEEKLY'
    MONTHLY = 'MONTHLY'

    __order__ = 'DAILY WEEKLY MONTHLY'


SCALES_INTERVALS = {
    Scales.DAILY: relativedelta(days=1),
    Scales.WEEKLY: relativedelta(weeks=1),
    Scales.MONTHLY: relativedelta(months=1)
}

# это понедельник, подходит для всех скейлов
# для отчётов, данные в которых не распределены по времени
COMMON_AGGREGATION_POINT = datetime(2001, 1, 1)


def get_keys(scale, start, end=datetime.now(MOSCOW_TZ)):
    """Строит временную шкалу от start до end для переданного масштаба.

    :param str scale: масштаб графика
    :param datetime start: начало координат по оси x
    :param datetime end: конечная точка оси x

    :rtype: list
    :return: список дат для оси x графика
    """
    start = start.astimezone(MOSCOW_TZ)
    end = end.astimezone(MOSCOW_TZ)
    current_point = get_aggregation_point(scale, start)
    result = []

    while current_point < end:
        result.append(current_point)
        current_point += SCALES_INTERVALS.get(Scales[scale])

    result.append(current_point)
    return result


def get_aggregation_point(scale, date):
    """Возвращает точку, в которую попадает переданная дата в данном масштабе.

    :param str scale: масштаб графика
    :param datetime date: дата, которую нужно агрегировать в какую-то точку

    :rtype: datetime
    :return:
        Для месячного масштаба - первый день месяца.
        Для недельного - дата ближайшего прошедшего понедельника.
        Для дневного - начало дня.
    """
    scale = Scales[scale]
    date = date.astimezone(MOSCOW_TZ)
    start_of_the_day = datetime(date.year, date.month, date.day, tzinfo=MOSCOW_TZ)

    if scale is Scales.DAILY:
        return start_of_the_day
    elif scale is Scales.WEEKLY:
        return start_of_the_day + relativedelta(weekday=MO(-1))
    elif scale is Scales.MONTHLY:
        return start_of_the_day.replace(day=1)


def get_start_point(scale, dates):
    """Ищет минимальную дату среди переданных, затем отступает назад на период, соответствующий масштабу графика.
    Отступать назад нужно для графиков, которые агрегируют точки по условию, включающему сравнение со следующей точкой.
    В таких графиках задачи, которые должны попасть в первую точку, не будут учтены, если не добавить точку перед ней.
    См. https://st.yandex-team.ru/FEI-10256

    :param str scale: масштаб графика
    :param list dates: список дат

    :rtype: datetime.datetime
    :return: начальная точка для графика, если не нашлась, то None
    """
    return get_aggregation_point(scale, min(dates)) - SCALES_INTERVALS.get(Scales[scale]) if dates else None


def fielddate(date):
    """Приводит дату к строке в формате для загрузки в Стат.

    :param datetime date: точка агрегации

    :rtype: str
    """
    return date.strftime('%Y-%m-%d')
