# -*- coding: utf-8 -*-

from enum import Enum

import sandbox.projects.infratools.vteam.libs.issues as sti
from sandbox.projects.infratools.vteam.libs.series.conditions import Condition


def from_type(data):
    """Возвращает экземпляр класса, описывающего условие поиска границ

    :param dict data: данные об условии поиска

    :rtype: Boundary
    """
    field_id = data.get('field')

    if field_id in ('start', 'end', 'createdAt', 'resolvedAt', 'updatedAt'):
        return ValueBoundary(data)
    else:
        return HistoryBoundary(data)


class HistorySearch(Enum):
    """Тип логики поиска изменений в истории задачи"""
    first = 'first'  # искать первое изменение поля
    last = 'last'  # искать последнее изменение поля
    current = 'current'  # дата изменения на текущее значение, если оно совпадает с требуемым
    strict_last = 'strict_last'  # искать последнее изменение поля без учета эквивалетных полей


class Boundary(object):
    """Класс для поиска даты, которая будет являться границей интервала для задачи.
    Этот интервал — значение задачи на линии графика.
    """
    field_id = None

    @property
    def description(self):
        raise NotImplementedError('Boundary description is not defined')

    def __init__(self, data):
        self.field_id = data.get('field')

    def of(self, issue):
        """Возвращает границу для переданной задачи, на основе параметров границы.

        :param dict issue: данные задачи

        :rtype: datetime.datetime
        """
        raise NotImplementedError()


class HistoryBoundary(Boundary):
    """Дата изменения поля в ченжлоге"""
    search_for = None  # искать первое или последнее изменение
    condition = None  # условие, по которому ищется нужное изменение

    @property
    def description(self):
        return '%s %s' % (
            {
                'first': u'первого',
                'last': u'последнего',
                'current': u'текущего',
                'strict_last': u'строго последнего',
            }.get(self.search_for.name),
            self.condition.description
        )

    def __init__(self, data):
        super(HistoryBoundary, self).__init__(data)

        self.condition = Condition(data)

        search_for = data.get('search_for', 'first')
        try:
            self.search_for = HistorySearch[search_for]
        except KeyError:
            raise TypeError('Unexpected history search type: "%s"' % search_for)

    def of(self, issue):
        if self.search_for is HistorySearch.first:
            return sti.first_change(issue, self.field_id, fn=self.condition.test)
        elif self.search_for is HistorySearch.last:
            return sti.last_change(issue, self.field_id, fn=self.condition.test)
        elif self.search_for is HistorySearch.current:
            return sti.last_change(issue, self.field_id, fn=self.condition.test, require_current=True)
        elif self.search_for is HistorySearch.strict_last:
            return sti.last_change(issue, self.field_id, fn=self.condition.test, stack_changes=False, require_current=True)


class ValueBoundary(Boundary):
    """Конкретное значение поля, в поле хранится дата"""

    @property
    def description(self):
        return self.field_id

    def of(self, issue):
        return sti.value(issue, self.field_id)
