# -*- coding: utf-8 -*-

from enum import Enum

import sandbox.projects.infratools.vteam.libs.issues as sti
import sandbox.projects.infratools.vteam.libs.operators as operator


def _parse(value, field_id):
    """Парсит значение, если необходимо.

    Из конфига значения приходят либо строкой, либо числом. Иногда это надо распарсить в сложный тип, чтобы
    тип значений совпадал с типом значений в данных задачи.

    :param value: значение
    :param str field_id: идентификатор поля в Стартреке
    """
    if field_id in ('start', 'end', 'createdAt', 'resolvedAt', 'updatedAt', 'spent'):
        return sti._parse_field(value, field_id)
    else:
        return value


def _stringify(value, field_id):
    """Приводит значение к строке для записи в измерение отчёта.

    :param value:
    :param str field_id: идентификатор поля в Стартреке

    :rtype: basestring
    """
    if field_id in ('spent',):
        return u'%d ч/ч' % (value.total_seconds() / 3600)
    else:
        return str(value)


class CompareTo(Enum):
    """Логика выбора значения из данных задачи для сравнения"""
    point = 'point'  # сравнивать со значением из истории, на момент даты текущей точки
    current = 'current'  # сравнивать с последним значением, вне зависимости от даты точки


class Condition(object):
    """Класс для проверки значения поля задачи на основе конфига"""
    field_id = None  # поле, по которому производится проверка. Нужно для парсинга значения, если оно не примитив
    values = ()  # набор целевых значений, с которыми нужно сравнивать переданное значение
    is_current_value = False  # сравнивать поле из changelog с текущим значением в тикете, игнорируется свойство values
    operation = None  # оператор условия
    fn = None  # функция проверки, для одного значения из списка
    compare_to = None  # какое значение поля брать из данных задачи для сравнения

    @property
    def description(self):
        if self.target_values:
            return '%s %s %s' % (
                self.field_id,
                {
                    'eq': '=',
                    'ne': u'≠',
                    'lt': '<',
                    'le': u'≤',
                    'gt': '>',
                    'ge': u'≥'
                }.get(self.operation, self.operation),
                ', '.join(_stringify(val, self.field_id) for val in self.target_values)
            )

        return self.field_id

    def __init__(self, data):
        self.field_id = data.get('field')
        self.is_current_value = data.get('is_current_value')
        self.target_values = tuple(_parse(value, self.field_id) for value in data.get('values', []))

        self.operation = data.get('operation', 'eq')
        try:
            self.fn = getattr(operator, self.operation)
        except AttributeError:
            raise TypeError('Unexpected condition operation: "%s"' % self.operation)

        compare_to = data.get('compare_to', 'point')
        try:
            self.compare_to = CompareTo[compare_to]
        except KeyError:
            raise TypeError('Unexpected value for <compare_to>: "%s"' % compare_to)

    def test(self, value):
        """Проверяет, удовлетворяет ли переданное значение условию из конфига

        :param value: значение для проверки

        :rtype: bool
        """
        # если целевых значений нет, условие выполнено при любом непустом значении
        if not self.target_values:
            return value is not None and value != ''

        # если сравнение порядковое, то при отсутствующем значении условие никогда не выполняется
        if self.operation in ('gt', 'ge', 'lt', 'le') and value is None:
            return False

        if isinstance(value, list):
            if self.operation in ('eq', 'ne', 'intersect', 'different', 'superset', 'subset'):
                return self.fn(value, self.target_values)
            raise NotImplementedError('Operation "%s" is not implemented for list' % self.operation)

        if value is None:
            if self.operation in ('intersect', 'different', 'superset', 'subset'):
                return self.fn([], self.target_values)

        lookup = any
        if self.operation == 'ne':
            lookup = all

        return lookup(self.fn(value, target) for target in self.target_values)

    def test_issue(self, issue, date=None):
        """Проверяет, удовлетворяет ли значение поля задачи условию из конфига.
        В зависимости от конфига, сравнивает последнее значение поля или значение в какой-то момент ченжлога.

        :param dict issue: данные задачи
        :param datetime.datetime date: дата, на момент которой надо взять значение поля задачи для сравнения

        :rtype: bool
        """
        if self.compare_to is CompareTo.point:
            values = sti.last_value(issue, self.field_id, before=date)
        else:
            values = sti.value(issue, self.field_id)

        if self.is_current_value:
            current_value = sti.value(issue, self.field_id)
            self.target_values = tuple(current_value if isinstance(current_value, list) else [current_value])

        return self.test(values)
