# -*- coding: utf-8 -*-

import numpy as np


def from_type(data):
    """Возвращает экземпляр нужной функции, в зависимости от поля `type`.

    :param dict data: данные о функции

    :rtype: SeriesFn

    :raises NotImplementedError: в случае некорректного типа
    """
    try:
        return {
            'percentile': Percentile,
            'average': Average
        }[data.get('type')](data)
    except KeyError as key:
        raise NotImplementedError('Series function "%s" is not implemented' % key)


class SeriesFn(object):
    """Функция для вычисления конечного значения в точке на графике"""
    type = None

    def __init__(self, data):
        self.type = data.get('type')

    @property
    def title(self):
        """Название функции для записи в качестве значения измерения отчёта"""
        raise NotImplementedError()

    def calc(self, values):
        """Вычисляет значение точки.

        :param list values: список значений, попавших в точку
        """
        raise NotImplementedError()


class Percentile(SeriesFn):
    """Проценитль. По договорённости, вычисление процентиля, попавшего между элементами, происходит интерполяцией вверх:
    если процентиль находится между точками i < j, выбирается j.
    См. https://st.yandex-team.ru/FEI-8662
    """
    threshold = None

    def __init__(self, data):
        super(Percentile, self).__init__(data)
        try:
            self.threshold = data['threshold']
        except KeyError:
            raise TypeError('Percentile threshold is not defined')

    @property
    def title(self):
        return u'%s процентиль' % self.threshold

    def calc(self, values):
        return np.percentile(values, self.threshold, interpolation='higher')


class Average(SeriesFn):
    title = u'Среднее'

    def calc(self, values):
        return np.average(values)
