# -*- coding: utf-8 -*-

import sandbox.projects.infratools.vteam.libs.issues as sti
from sandbox.projects.infratools.vteam.libs.utils import parse_date
from sandbox.projects.infratools.vteam.libs.series.fn import from_type as fn_from_type
from sandbox.projects.infratools.vteam.libs.series.boundaries import from_type as boundary_from_type
from sandbox.projects.infratools.vteam.libs.series.conditions import Condition


class Series(object):
    """Описывает логику построения линии на графике"""
    id = None  # ID линии
    title = None  # лэйбл линии
    fn = None  # функции для расчёта значения в точке; для каждой функции — отдельная линия
    conditions = []  # условия, определяющие, попадает ли задача в линию
    goal = None  # значение целевой линии
    position = None  # позиция линии на графике (в т. ч. определяет цвет линии)

    @property
    def description(self):
        raise NotImplementedError('Series description is not defined')

    def __init__(self, item):
        self.id = item.get('id')
        self.title = item.get('title')
        self.fn = [fn_from_type(fn_data) for fn_data in item.get('fn')]
        self.conditions = [Condition(condition) for condition in item.get('conditions', [])]
        self.goal = item.get('goal')
        self.position = item.get('position')

    def includes(self, issue, date):
        """Попадает ли задача в линию графика. Проверяются условия из конфига. Если условий несколько,
        они должны выполняться все. Если условий нет, задача в линию автоматически попадает.

        :param dict issue:
        :param datetime.datetime date:

        :rtype: bool
        """
        return all(condition.test_issue(issue, date) for condition in self.conditions)


class SlaSeries(Series):
    @property
    def description(self):
        pass

    def __init__(self, item):
        super(SlaSeries, self).__init__(item)
        self.sla_id = item.get('sla_id')

    def _sla(self, issue):
        sla_list = sti.value(issue, 'sla')

        if not sla_list:
            raise ValueError('Issue %s expected to have an SLA with ID %s' % (sti.value(issue, 'key'), self.sla_id))

        try:
            return next(sla for sla in sla_list if sla.get('settingsId') == self.sla_id)
        except StopIteration:
            raise ValueError('Issue %s expected to have an SLA with ID %s' % (sti.value(issue, 'key'), self.sla_id))

    def value(self, issue):
        sla = self._sla(issue)
        threshold = sla.get('failedThreshold') or 0
        remaining = sla.get('toFailTimeWorkDuration') or threshold
        return threshold - remaining

    def sla_goal(self, issue):
        return self._sla(issue).get('failedThreshold')

    def includes(self, issue, date):
        matches_conditions = super(SlaSeries, self).includes(issue, date)
        sla_start, sla_end = self.boundaries(issue)
        return matches_conditions and sla_start is not None and sla_start < date

    def boundaries(self, issue):
        sla = self._sla(issue)
        sla_start = parse_date(sla.get('startedAt'))
        sla_end = parse_date(sla.get('pausedAt')) if sla.get('clockStatus') == 'STOPPED' else None
        return sla_start, sla_end


class IntervalSeries(Series):
    """В интервальных линиях значение для задачи — это промежуток времени между двумя условиями"""
    start = None
    end = None

    @property
    def description(self):
        return u'От %s до %s' % (self.start.description, self.end.description)

    def __init__(self, item):
        super(IntervalSeries, self).__init__(item)
        self.start = boundary_from_type(item.get('start'))
        self.end = boundary_from_type(item.get('end'))

    def boundaries(self, issue):
        return self.start.of(issue), self.end.of(issue)


class IntervalSeriesMulti(Series):
    """В интервальных линиях значение для задачи — это промежуток времени между несколькими условиями"""
    start = None
    end = None

    @property
    def description(self):
        return u'От %s до %s' % (self.start.description, self.end.description)

    def __init__(self, item):
        super(IntervalSeriesMulti, self).__init__(item)
        self.start = [boundary_from_type(start_item) for start_item in item.get('start')]
        self.end = [boundary_from_type(end_item) for end_item in item.get('end')]

    def boundaries(self, issue):
        end_dates = filter(bool, [end_item.of(issue) for end_item in self.end])
        end_date = min(end_dates) if len(end_dates) else None

        start_dates = filter(bool, [start_item.of(issue) for start_item in self.start])

        if end_date:
            start_dates = filter(lambda start_date: start_date < end_date, start_dates)

        start_date = max(start_dates) if len(start_dates) else None

        return start_date, end_date
