# -*- coding: utf-8 -*-

from dateutil import parser as dateparser
from datetime import timedelta

from .worktime import MOSCOW_TZ


def leading_num(string, default=None):
    """Парсит число в начале строки, отбрасывая любую нечисловую последовательность символов, стоящую дальше

    :param basestring string:
    :param float, None default:
    :rtype: float, None
    """
    try:
        return float(string)
    except TypeError as e:
        if string is None:
            return default
        else:
            raise e
    except ValueError:
        res = ''
        for char in string.strip():
            if char.isdigit():
                res += char
            elif char in '.,' and '.' not in res:
                res += '.'
            else:
                break
        return float(res) if len(res) and res != '.' else default


def parse_date(date):
    """Парсит дату

    :param str date:

    :rtype: datetime.datetime
    :return: распаршенная дата

    :raises ValueError: если значение не получается распарсить как дату
    """
    if not date:
        return None

    try:
        date = dateparser.parse(date)
    except ValueError:
        raise ValueError('Unknown date format: "%s"' % date)

    try:
        date = date.astimezone(MOSCOW_TZ)
    except ValueError:  # naive datetime
        date = MOSCOW_TZ.localize(date)

    return date


def parse_iso_duration(iso_duration):
    """Переводит ISO 8601 duration в timedelta.
    Особенность формата в Стартреке: используются интервалы рабочих дней, т. е.

    - 1 день = 8 часов
    - 1 неделя = 5 дней = 40 часов

    Интервалы крупнее недели не используются из-за неоднозначности.
    См. https://st.yandex-team.ru/TOOLSUP-35805

    :param str iso_duration:

    :rtype: timedelta
    """
    if not iso_duration:
        return None

    date, _, time = iso_duration[1:].partition('T')

    weeks, _, date = date.rpartition('W')
    days, _, _ = date.rpartition('D')

    hours, _, time = time.rpartition('H')
    minutes, _, time = time.rpartition('M')
    seconds, _, _ = time.rpartition('S')

    weeks = int(weeks or 0)
    days = int(days or 0)

    hours = int(hours or 0) + days * 8 + weeks * 40
    minutes = int(minutes or 0)
    seconds = int(seconds or 0)

    return timedelta(hours=hours, minutes=minutes, seconds=seconds)


def russian_date(date, fmt='d MMMM yyyy'):
    """Приводит даты к человекочитаемому формату в русской локали.

    :param datetime.datetime date:
    :param str fmt:

    :rtype: str
    """
    from babel.dates import format_date

    return format_date(date, format=fmt, locale='ru') if date else None
