# -*- coding: utf-8 -*-

import pytz
import numpy as np
from datetime import datetime, timedelta

MOSCOW_TZ = pytz.timezone('Europe/Moscow')


def worktime(start, end, exclude_holidays=True):
    """Вычисляет таймдельту между датами, исключая выходные.

    :param datetime start:
    :param datetime end:
    :param bool exclude_holidays:

    :rtype: timedelta
    """
    if not exclude_holidays:
        return end - start

    start_day = start.date()
    end_day = end.date()

    if not np.is_busday(start_day):
        start_day = np.busday_offset(start_day, 0, roll='forward')
        start = start_day.astype('datetime64[s]').astype(datetime).replace(tzinfo=MOSCOW_TZ)

    if not np.is_busday(end_day):
        # weekmask включает субботу, чтобы `end` стал 00:00 субботы, и пятница учлась целиком
        end_day = np.busday_offset(end_day, 0, roll='backward', weekmask='Mon Tue Wed Thu Fri Sat')
        end = end_day.astype('datetime64[s]').astype(datetime).replace(tzinfo=MOSCOW_TZ)

    weekends = np.busday_count(start_day, end_day, weekmask='Sat Sun')

    return end - start - timedelta(days=weekends)


def bound(start, end, lower=None, upper=None):
    """Ограничивает интервал дат другим интервалом. Итоговые значения будут не больше end.

    :param datetime start:
    :param datetime end:
    :param datetime lower:
    :param datetime upper:

    :return: tuple
    """
    if lower is not None:
        start = max(lower, start)

    if upper is not None:
        end = min(upper, end)

    if start > end:
        start = end

    return start, end
