# -*- coding: utf-8 -*-
import copy
import json
import logging
import math
import sys
from collections import OrderedDict
from datetime import date, timedelta
from datetime import datetime
from decimal import Decimal
from itertools import chain

import requests
import sandbox.sdk2 as sdk2

from sandbox.projects.common import binary_task
from sandbox.projects.inventori.common.resources import InventoriBackToBackReportFile
from sandbox.projects.inventori.common.resources import ROBOT_INVENTORI_SECRET

if sys.version_info[:2] < (3, 0):
    text = unicode
    text_types = (unicode, str)
else:
    text = str
    text_types = (str,)

# # Sorry for this dummy, stupid code
# # I promise that i come back and refactor it
# # Thank u for your understanding
#
# NP, I'm here to help you!

DEBUG_RESPONSE_A = '__debug_response_a__'
DEBUG_RESPONSE_B = '__debug_response_b__'

LOGGER_NAME = 'inventori backtoback testing'
logger = logging.getLogger(LOGGER_NAME)

EMPTY_MARK = "∅"
EQUAL_MARK = ""
UNMEASURED_DIFF = r"¯\_(ツ)_/¯"
INF_MARK = 'over 9000'

WHOLE_JSON_DIFF = "__whole_json_diff__"

QUERY_GET_AMMUNITION = '''PRAGMA yt.InferSchema;
PRAGMA yt.TmpFolder = "//home/inventori/tmp";
PRAGMA AnsiInForEmptyOrNullableItemsCollections;

SELECT
    x.*,
    TableRecordIndex() AS id,
FROM
    `{yt_ammunition_path}` AS x
;'''

DETAILS = """<details>
    <summary class="{details_class}">TEST #{test_number} | {description} |  id=[{id}]</summary>
    <div class="row">
      <table class="info-table table table-bordered table-hover" style="width: 100%;">
        <tbody>
          <tr>
            <th>field_name</th>
            <th>{address_one}</th>
            <th>difference</th>
            <th>{address_two}</th>
          </tr>
          {rows}
        </tbody>
      </table>
    </div>
    <hr width=100%>
    <div class="requestUrl">{request_url}</div>
    <div class="requestBody">{request_body}</div>
    <div class="requestBody">{request_body_original}</div>
</details>"""

HTML_CONTENT = """<!DOCTYPE html PUBLIC "-//W3C//DTD HTML 4.01//EN" "http://www.w3.org/TR/html4/strict.dtd">
        <html>
        <head>
        <style>
        .column {
            float: left;
            width: 33.33%;
            min-height: 150px;
        }
        .row:after {
            content: "";
            display: table;
            clear: both;
            border-color: #000000;
            border:1px solid black;
        }
        .requestUrl{
            clear: left;
            background: #ccc;
            border-color: #000000;
            border:1px solid black;
            min-height: 20px;
        }
        .requestBody{
            clear: left;
            background: #ccc;
            border-color: #000000;
            border:1px solid black;
            min-height: 50px;
        }
        .redMarked{
          background: #ffc0cb;
        }
        .greenMarked{
          background: #98fb98;
        }
        .yellowMarked{
          background: #FFED4F;
        }
        .summaryError{
            background:   #ff0000;
        }
        .summaryError100{
            background-color: #FF3333;
        }
        .summaryError80{
            background-color: #FF5B5B;
        }
        .summaryError60{
            background-color: #FF7B7B;
        }
        .summaryError40{
            background-color: #FF8888;
        }
        .summaryError20{
            background-color: #FF9B9B;
        }
        .summaryOk{
            background:   #008000;
        }
        .periodsSplitter{
            background:   #afeeee
        }
        .info-table td {
            text-align: center;
        }
        </style>
        </head>
        <body>"""

TAIL = """
        <div>{counter_message}</div>
        {details_content}
        </body>
        </html>
            """

PERIODS_SPLITTER = '<div class="periodsSplitter">|    {start_date} - {end_date}    |     {period_in_days} days</div>'
NON_PERIOD_SPLITTER = '<div class="periodsSplitter">|    Non period queries    |</div>'

HTTP_PREFIX = "http://"

TABLE_ATTRIBUTES = 'id="info-table" class="table table-bordered table-hover"'

DIFF_TABLE = """<div class="row">
  <table id="info-table" class="table table-bordered table-hover" style="width: 100%;">
    <tbody>
      <tr>
        <th>field_name</th>
        <th>{ethalon_server_name}</th>
        <th>difference</th>
        <th>{dev_server_name}</th>
      </tr>
      {rows}
    </tbody>
  </table>
</div>"""

DIFF_TABLE_ROW = """<tr class="{cls}">
    <th>{field_name}</th>
    <td>{ethalon_value}</td>
    <td>{difference}</td>
    <td>{dev_value}</td>
</tr>"""


def fmt_value(value):
    if isinstance(value, (int, float)):
        return "{:,}".format(value).replace(",", "_")
    return value


def fmt_abs_diff(diff):
    if int(diff) == diff:
        return "{:,}".format(diff).replace(",", "_")
    return "{:,.4f}".format(diff).replace(",", "_")


def get_precision(value):
    """
    :type value: Decimal
    :rtype: int
    """
    value_int = Decimal(int(value))
    fractional_part = value - value_int
    if fractional_part == 0:
        return 0
    n = 1
    while fractional_part % Decimal('0.1') != 0:
        n += 1
        fractional_part /= Decimal('0.1')
    return n


def get_significant(value):
    """
    :type value: int | float | Decimal
    :rtype: int
    """
    n = 0
    while int(value) == 0:
        n += 1
        value *= 10
    return n


class YqlErrorException(Exception):
    pass


def make_json_objects_sorted(json):
    """
    :type json: dict | list | string | int | float
    :rtype: dict | list | string | int | float

    >>> make_json_objects_sorted(OrderedDict([('title', 'Dune'),
    ...     ('author', {'name': 'Frank', 'surname': 'Herbert'})]))
    OrderedDict([('author', OrderedDict([('name', 'Frank'), ('surname', 'Herbert')])), ('title', 'Dune')])

    """
    if hasattr(json, "items"):
        return OrderedDict(sorted(
            [(k, make_json_objects_sorted(v)) for k, v in json.items()],
            key=lambda x: x[0]
        ))
    if not isinstance(json, text_types) and hasattr(json, "__iter__"):
        return [make_json_objects_sorted(x) for x in json]
    return json


def make_json_flat(json, prefix=""):
    """Make json flat object if possible

    :type json: dict | list | string | int | float
    :type prefix: str
    :rtype: OrderedDict | list | string | int | float

    >>> make_json_flat(OrderedDict([('title', 'Dune'),
    ...    ('author', OrderedDict([('name', 'Frank'), ('surname', 'Herbert')]))]))
    OrderedDict([('title', 'Dune'), ('author.name', 'Frank'), ('author.surname', 'Herbert')])

    >>> make_json_flat(OrderedDict([('title', 'Dune'), ('houses', ['Atreides', 'Harkonnen'])]))
    OrderedDict([('title', 'Dune'), ('houses.0', 'Atreides'), ('houses.1', 'Harkonnen')])

    >>> make_json_flat(['Dune', 'Dune Messiah', 'Children of Dune'])
    OrderedDict([('0', 'Dune'), ('1', 'Dune Messiah'), ('2', 'Children of Dune')])

    >>> make_json_flat('Dune')
    'Dune'
    """
    def do_prefix(field_name):
        return "{}.{}".format(prefix, field_name) if prefix else str(field_name)
    if hasattr(json, "items"):
        return OrderedDict(chain(*(make_json_flat(v, prefix=do_prefix(k)).items() for k, v in json.items())))
    if not isinstance(json, text_types) and hasattr(json, "__iter__"):
        return OrderedDict(chain(*(make_json_flat(v, prefix=do_prefix(n)).items() for n, v in enumerate(json))))
    return OrderedDict([(prefix, json)]) if prefix else json


def json2html_convert(json, sorted=True, flat=False):
    from json2html import json2html
    json = make_json_objects_sorted(json) if sorted else json
    json = make_json_flat(json) if sorted else json
    return json2html.convert(json=json, table_attributes=TABLE_ATTRIBUTES)


class InventoriBackToBackTestingTask(binary_task.LastBinaryTaskRelease, sdk2.Task):
    count = 0
    result_of_tests = ""
    error_count = 0

    class Requirements(sdk2.Requirements):
        disk_space = 80 * 1024  # 80 Gb for data (all necessary indices + archive)
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        ext_params = binary_task.binary_release_parameters_list(stable=True)

        yt_yql_cluster = sdk2.parameters.String("YT cluster", required=True, default="arnold")

        yql_token = sdk2.parameters.YavSecretWithKey(
            "YQL token",
            default="#".join((ROBOT_INVENTORI_SECRET, "yql_token")),
            required=True
        )

        with sdk2.parameters.Group("Aim") as aim_params:

            ethalon_env = sdk2.parameters.String("Ethalon server", required=True, default="inventori.yandex-team.ru")

            dev_env = sdk2.parameters.String("Dev server", required=True, default="inventori-test.yandex-team.ru")

            diff_threshold = sdk2.parameters.String("Threshold (in %) to count values with less diff as equal", required=True, default=0.7)

            field_diff_threshold = sdk2.parameters.Dict("Threshold for specific response fields")

        with sdk2.parameters.Group("Ammo") as ammo_params:

            yt_ammunition_path = sdk2.parameters.String("Load ammo from", required=True)
            bullet_description_field_name = sdk2.parameters.String(
                "Bullet description field name", default="description", required=True)
            request_payload_field_name = sdk2.parameters.String(
                "Request payload field name", default="message_body", required=True)
            api_endpoint_path_field_name = sdk2.parameters.String(
                "Api endpoint path field name", default="uri", required=True)
            magic_string = sdk2.parameters.String(
                "Magic string - (offset:periodInDays,etc ) 0:7",
                description="Used for ammo start_date and end_date change."
                            " A:B means A - offset in days from now, B - period in days",
                required=True,
                default="0:7,0:10,180:7")
            fail_fast_string = sdk2.parameters.String("Failfast List", required=False)

        with sdk2.parameters.Group("Report") as report_params:

            startrek_token = sdk2.parameters.YavSecretWithKey(
                "Startrek key vault name",
                default="#".join((ROBOT_INVENTORI_SECRET, "startrek_token")),
                required=True,
            )

            startrek_ticket = sdk2.parameters.String("Startrek ticket", required=True)

    def process_bullet(self, bullet, ethalon_env, dev_env, start_date=None, end_date=None):
        assert (start_date is None and end_date is None) or (start_date and end_date)
        from jsondiff import diff
        from jsondiff import symbols

        self.count += 1
        bullet_id = str(bullet["id"])
        if DEBUG_RESPONSE_A in bullet and DEBUG_RESPONSE_B in bullet:
            response_one = bullet[DEBUG_RESPONSE_A]
            response_two = bullet[DEBUG_RESPONSE_B]
            json_body = None
            request_url = ""
            description = bullet[self.Parameters.bullet_description_field_name]
        else:
            request_url = bullet[self.Parameters.api_endpoint_path_field_name]
            request_body = bullet[self.Parameters.request_payload_field_name]
            description = bullet[self.Parameters.bullet_description_field_name]
            bullet_id = str(bullet["id"])

            ethalon_url = HTTP_PREFIX + ethalon_env + request_url
            dev_url = HTTP_PREFIX + dev_env + request_url
            if request_body:
                try:
                    json_body = json.loads(request_body)
                except json.decoder.JSONDecodeError:
                    logger.error("bad request body %s", request_body)
                    raise

                logger.info("--- request_body: %s", request_body)

                if start_date and end_date:
                    json_body["campaign_parameters"]["strategy"]["start_date"] = str(start_date)
                    json_body["campaign_parameters"]["strategy"]["end_date"] = str(end_date)

                logger.info("POST {%s} {%s}", ethalon_url, json_body)
                result_one = requests.post(url=ethalon_url, json=json_body)
                logger.info("POST {%s} {%s}", dev_url, json_body)
                result_two = requests.post(url=dev_url, json=json_body)
            else:
                json_body = None
                logger.info("GET {%s}", ethalon_url)
                result_one = requests.get(url=ethalon_url)
                logger.info("GET {%s}", dev_url)
                result_two = requests.get(url=dev_url)

            response_one = result_one.json()
            response_two = result_two.json()

        response_json_one = make_json_objects_sorted(make_json_flat(response_one))
        response_json_two = make_json_objects_sorted(make_json_flat(response_two))

        logger.info("--- response from ethalon: %s", response_json_one)
        logger.info("--- response from test stand: %s", response_json_two)

        if json_body:
            html_request_body = json2html_convert(json_body)
        else:
            html_request_body = ""

        def measure_diff(one, two):
            if not isinstance(one, (int, float)) or not isinstance(two, (int, float)):
                return None
            abs_diff = abs(one - two)
            float_abs_diff = float(abs_diff)
            float_diff_value = float(one)
            if float_diff_value == 0:
                percent_diff = INF_MARK
            else:
                percent_diff = (float_abs_diff / float_diff_value) * 100 if float_diff_value > 0.0001 else 0
            return abs_diff, percent_diff

        def process_difference(diff, on_insert, on_delete, on_change):
            if isinstance(diff, list):
                one, two = copy.deepcopy(diff)
                if isinstance(one, dict) and isinstance(two, dict):
                    for k, v in one.items():
                        logging.debug("on_delete one %s", one)
                        on_delete(k, v)
                    for k, v in two.items():
                        logging.debug("on_insert two %s", two)
                        on_insert(k, v)
                else:
                    on_change(WHOLE_JSON_DIFF, *diff)
            else:
                for diff_element, diff_value in diff.items():
                    if diff_element == symbols.insert:
                        for k, v in diff_value.items():
                            on_insert(k, v)
                    elif diff_element == symbols.delete:
                        for k, v in diff_value.items():
                            on_delete(k, v)
                    elif hasattr(diff_value, "items"):
                        raise ValueError("Objects must be flattered before checking for diffs")
                    else:
                        on_change(diff_element, *diff_value)

        if hasattr(response_json_one, "items"):
            diff_json = {k: EQUAL_MARK for k in response_json_one}
        elif not isinstance(response_json_one, text_types) and hasattr(response_json_one, "__iter__"):
            diff_json = {str(n): EQUAL_MARK for n in range(len(response_json_one))}
        elif response_json_one == response_json_two:
            diff_json = {WHOLE_JSON_DIFF: EQUAL_MARK}
        else:
            diff_json = {}

        tr_class = {}

        diff_stat = {
            "max_diff_percent": 0,
            "unmeasured_diff": 0,
            "has_diff": False,
        }
        rows_html = ""

        details_class = "summaryOk"
        if response_json_one != response_json_two:
            difference = diff(response_json_one, response_json_two, syntax="symmetric")
            logger.info("difference = %s", difference)

            def on_insert_1(name, _):
                tr_class[name] = "greenMarked"
                diff_json[name] = UNMEASURED_DIFF
                logging.debug("set %s for %s", EMPTY_MARK, name)
                response_json_one[name] = EMPTY_MARK
                diff_stat["unmeasured_diff"] += 1
                diff_stat["has_diff"] = True

            def on_delete_1(name, _):
                tr_class[name] = "redMarked"
                diff_json[name] = UNMEASURED_DIFF
                response_json_two[name] = EMPTY_MARK
                logging.debug("set %s for %s", EMPTY_MARK, name)
                diff_stat["unmeasured_diff"] += 1
                diff_stat["has_diff"] = True

            def on_change_1(name, value_a, value_b):
                diff_measures = measure_diff(value_a, value_b)
                if diff_measures:
                    abs_diff, percent_diff = diff_measures
                    diff_threshold = Decimal(self.Parameters.field_diff_threshold.get(
                        name, self.Parameters.diff_threshold))
                    logging.debug("diff_threshold for field %s is %s", name, diff_threshold)
                    logging.debug(
                        "abs_diff <= diff_threshold is %s <= %s is %s",
                        abs_diff, diff_threshold, abs_diff <= diff_threshold)
                    if percent_diff == INF_MARK:
                        diff_json[name] = "{} : {}%".format(fmt_abs_diff(abs_diff), percent_diff)
                        diff_stat["max_diff_percent"] = max(diff_stat["max_diff_percent"], float('inf'))
                        logging.debug(
                            "has inf diff for field %s where %s != %s abs_diff=%s percent_diff=%s",
                            name, value_a, value_b, abs_diff, percent_diff)
                        diff_stat["has_diff"] = True
                        tr_class[name] = "yellowMarked"
                        return
                    elif percent_diff <= diff_threshold:
                        diff_stat["max_diff_percent"] = max(diff_stat["max_diff_percent"], percent_diff)
                        diff_json[name] = "{0} : {1:.{2}f}%".format(
                            fmt_abs_diff(abs_diff), percent_diff,
                            max(get_precision(diff_threshold), get_significant(percent_diff)))
                        return
                    diff_json[name] = "{0} : {1:.{2}f}%".format(
                        fmt_abs_diff(abs_diff), percent_diff, get_precision(diff_threshold))
                    diff_stat["max_diff_percent"] = max(diff_stat["max_diff_percent"], percent_diff)
                    logging.debug(
                        "has diff for field %s where %s != %s abs_diff=%s percent_diff=%s",
                        name, value_a, value_b, abs_diff, percent_diff)
                    diff_stat["has_diff"] = True
                    tr_class[name] = "yellowMarked"
                else:
                    diff_json[name] = UNMEASURED_DIFF
                    diff_stat["unmeasured_diff"] += 1
                    logging.debug("has unmeasured for field %s where %s != %s", name, value_a, value_b)
                    diff_stat["has_diff"] = True
                    tr_class[name] = "yellowMarked"

            process_difference(difference, on_insert=on_insert_1, on_delete=on_delete_1,
                               on_change=on_change_1)

            logging.info("Process bullet %s with stat %s", bullet_id, diff_stat)
            if diff_stat["has_diff"]:
                self.error_count += 1
                details_class = "summaryError"
                if diff_stat["unmeasured_diff"] == 0:
                    max_diff_percent = diff_stat["max_diff_percent"]
                    details_class += str(int(math.ceil(float(min(100, max_diff_percent)) / 20) * 20))

        if WHOLE_JSON_DIFF in diff_json:
            rows_html += DIFF_TABLE_ROW.format(
                cls=tr_class.get(WHOLE_JSON_DIFF, ""),
                field_name="",
                ethalon_value=fmt_value(response_json_one),
                difference=diff_json[WHOLE_JSON_DIFF],
                dev_value=fmt_value(response_json_two),
            )
        else:
            for k, v in diff_json.items():
                rows_html += DIFF_TABLE_ROW.format(
                    cls=tr_class.get(k, ""),
                    field_name=k,
                    ethalon_value=fmt_value(response_json_one[k]),
                    difference=v,
                    dev_value=fmt_value(response_json_two[k]),
                )

        result_details = DETAILS.format(
            details_class=details_class,
            test_number=self.count,
            request_url=request_url,
            request_body=html_request_body,
            request_body_original=json.dumps(json_body, sort_keys=True),
            rows=rows_html,
            address_one=ethalon_env,
            address_two=dev_env,
            description=description,
            id=bullet_id,
        )
        logger.info(result_details)
        self.result_of_tests += result_details

    def send_result_to_startrek(self, startrek_ticket, startrek_token, resource, ethalon_env, dev_env, fail_fast_list):
        if not startrek_ticket:
            return

        from startrek_client import Startrek

        client = Startrek(useragent="python", token=startrek_token)
        issue = client.issues[startrek_ticket]

        if self.error_count > 0:
            if fail_fast_list:
                report_details = (
                    "Тесты бежали по короткому списку {fail_fast_list},"
                    " но, как ты понимаешь, они попадали."
                    " {error_count} ошибок из {shots_fired} тестов.".format(
                        fail_fast_list=fail_fast_list,
                        error_count=self.error_count,
                        shots_fired=self.count,
                    )
                )
            else:
                if ethalon_env == dev_env:
                    report_details = (
                        "Стрелял по-македонски по одному серверу! Но всё равно промахнулся."
                        " Смотри в отчет... что тут еще скажешь?"
                    )
                else:
                    report_details = (
                        "В репорт загляни Боярин! Что-то там не чисто!"
                        " {error_count} ошибок из {shots_fired} тестов.".format(
                            error_count=self.error_count,
                            shots_fired=self.count,
                        )
                    )
        else:
            if fail_fast_list:
                report_details = (
                    "Fail Fast список прошел без ошибок, но погоди радоваться -"
                    " дождись полного отчета."
                )
            else:
                if ethalon_env == dev_env:
                    report_details = (
                        "Стрелял по-македонски по одному серверу!"
                        " Неудивительно, что результаты сходятся!"
                    )
                else:
                    report_details = "Красава! ни одной ошибки! можешь катить!"

        comment = issue.comments.create(
            text="Координаты ударной батареи:"
            " https://sandbox.yandex-team.ru/task/{task_id}/view"
            "\n{report_details}"
            "\nОтчет: {report_url}".format(
                task_id=self.id,
                report_details=report_details,
                report_url=str(resource.http_proxy),
            ))
        self.set_info(
            "Ударная батарея репортует командуещему (￣^￣)ゞ:"
            ' <a href="https://st.yandex-team.ru/{startrek_ticket}#{comment_id}">'
            "https://st.yandex-team.ru/{startrek_ticket}#{comment_id}</a>".format(
                startrek_ticket=startrek_ticket,
                comment_id=comment.longId,
            ),
            do_escape=False,
        )

    def create_resource(self, result_html_content, report_name):
        resource = InventoriBackToBackReportFile(self, "Inventori BackToBack report", report_name)
        report_data = sdk2.ResourceData(resource)

        report_file_path = str(report_data.path)

        with open(report_file_path, 'w') as report_file:
            report_file.write(result_html_content)

        self.set_info(
            "Отчет об обстреле:"
            ' <a href="{report_url}">'
            "{report_url}</a>".format(
                report_url=str(resource.http_proxy),
            ),
            do_escape=False,
        )
        logging.info('Report file is ready: %s', report_file_path)
        report_data.ready()
        return resource

    def load_ammunition_from_yt(self, yql_proxy, yql_token, get_ammunition_query):
        from yql.api.v1.client import YqlClient
        from yql.client.operation import YqlOperationShareIdRequest

        def operation_callback(operation):
            share_request = YqlOperationShareIdRequest(operation.operation_id)
            share_request.run()
            share_url = "https://yql.yandex-team.ru/Operations/{id}".format(id=share_request.json)
            logger.info("--- YQL shared link on cluster (%s): %s", "arnold", share_url)

        yql_client = YqlClient(db=yql_proxy, token=yql_token)
        request = yql_client.query(get_ammunition_query, syntax_version=1)
        logger.info("YQL request started")
        logger.info("YQL QUERY:\n{query_str}".format(query_str=get_ammunition_query))
        start_time = datetime.now()
        request.run(pre_start_callback=operation_callback)
        results = request.get_results()
        duration = datetime.now() - start_time

        if not results.is_success:
            request.abort()
            if results.errors:
                for error in results.errors:
                    logger.error(" -- ERROR: " + error.format_issue())
                    raise YqlErrorException("Error during call to yt")
        else:
            logger.info("YQL request finished successfully, duration={duration}".format(
                duration=duration.total_seconds()))
        return [
            dict(zip(table.column_names, row))
            for table in results
            for row in table.get_iterator()
        ]

    def is_time_bullet(self, bullet):
        if (
            self.Parameters.request_payload_field_name not in bullet
            or not bullet[self.Parameters.request_payload_field_name]
        ):
            return False
        try:
            request_json = json.loads(bullet[self.Parameters.request_payload_field_name])
            return (
                isinstance(request_json, dict)
                and "campaign_parameters" in request_json
                and isinstance(request_json["campaign_parameters"], dict)
                and "strategy" in request_json["campaign_parameters"]
                and isinstance(request_json["campaign_parameters"]["strategy"], dict)
                and "start_date" in request_json["campaign_parameters"]["strategy"]
                and "end_date" in request_json["campaign_parameters"]["strategy"]
            )
        except json.JSONDecodeError:
            return False

    def process_all(self, ammunition, magic_string, ethalon_env, dev_env, startrek_ticket,
                    startrek_token, fail_fast_list=None):
        counter_message = "Ok tests {ok_count} of {all_count}. "
        failed_counter = "Failed tests {failed_count} of {all_count}"

        self.count = 0
        self.error_count = 0
        list_of_periods = magic_string.split(",")
        non_time_bullets = list(filter(lambda b: not self.is_time_bullet(b), ammunition))
        if non_time_bullets:
            self.result_of_tests += NON_PERIOD_SPLITTER
        for bullet in non_time_bullets:
                self.process_bullet(
                    bullet=bullet,
                    ethalon_env=ethalon_env,
                    dev_env=dev_env,
                )
        for element in list_of_periods:
            time_bullets = list(filter(self.is_time_bullet, ammunition))
            if not time_bullets:
                continue
            offset, period_in_days = map(int, element.split(":"))
            start_date = date.today() + timedelta(days=offset)
            end_date = start_date + timedelta(days=period_in_days - 1)
            period_in_days = (end_date - start_date).days + 1
            self.result_of_tests += PERIODS_SPLITTER.format(
                start_date=start_date,
                end_date=end_date,
                period_in_days=period_in_days,
            )
            for bullet in time_bullets:
                if fail_fast_list:
                    if bullet[self.Parameters.api_endpoint_path_field_name] in fail_fast_list:
                        self.process_bullet(
                            bullet=bullet,
                            ethalon_env=ethalon_env,
                            dev_env=dev_env,
                            start_date=start_date,
                            end_date=end_date,
                        )
                    else:
                        logger.info("Skipped for failfast id= %s", bullet[self.Parameters.api_endpoint_path_field_name])
                else:
                    self.process_bullet(
                        bullet=bullet,
                        ethalon_env=ethalon_env,
                        dev_env=dev_env,
                        start_date=start_date,
                        end_date=end_date,
                    )

        result_counter_message = counter_message.format(
            ok_count=self.count - self.error_count,
            all_count=self.count,
        )
        if self.error_count > 0:
            result_counter_message += failed_counter.format(
                failed_count=self.error_count,
                all_count=self.count,
            )
        result_html_content = HTML_CONTENT + TAIL.format(counter_message=result_counter_message,
                                                         details_content=self.result_of_tests)
        logger.info(result_html_content)
        if fail_fast_list:
            report_name = "report_back_to_back_fail_fast_list.html"
        else:
            report_name = "report_back_to_back.html"
        resource = self.create_resource(result_html_content=result_html_content, report_name=report_name)
        self.send_result_to_startrek(startrek_ticket=startrek_ticket,
                                     startrek_token=startrek_token,
                                     ethalon_env=ethalon_env,
                                     dev_env=dev_env,
                                     resource=resource,
                                     fail_fast_list=fail_fast_list)

    def on_execute(self):
        logger.info("Inventori Back to Back Testing Sandbox Task Started ...")

        yql_proxy = self.Parameters.yt_yql_cluster
        yql_token = self.Parameters.yql_token.data()[self.Parameters.yql_token.default_key]

        ethalon_env = self.Parameters.ethalon_env
        dev_env = self.Parameters.dev_env
        yt_ammunition_path = self.Parameters.yt_ammunition_path
        startrek_ticket = self.Parameters.startrek_ticket
        startrek_token = self.Parameters.startrek_token.data()[self.Parameters.startrek_token.default_key]
        magic_string = self.Parameters.magic_string
        fail_fast_string = self.Parameters.fail_fast_string

        get_ammunition_query = QUERY_GET_AMMUNITION.format(yt_ammunition_path=yt_ammunition_path)

        ammunition = self.load_ammunition_from_yt(
            yql_proxy=yql_proxy,
            yql_token=yql_token,
            get_ammunition_query=get_ammunition_query,
        )

        logging.debug("ammunition: %s", ammunition)

        if fail_fast_string:
            fail_fast_list = fail_fast_string.split(',')
            logger.info("failfast:  %s" % fail_fast_string)
            self.process_all(ammunition=ammunition,
                             magic_string=str(magic_string),
                             ethalon_env=ethalon_env,
                             dev_env=dev_env,
                             startrek_ticket=startrek_ticket,
                             startrek_token=startrek_token,
                             fail_fast_list=fail_fast_list)
        self.result_of_tests = ""
        if self.error_count == 0:
            self.process_all(ammunition=ammunition,
                             magic_string=str(magic_string),
                             ethalon_env=ethalon_env,
                             dev_env=dev_env,
                             startrek_ticket=startrek_ticket,
                             startrek_token=startrek_token)

        logger.info("Inventori BackToBack Testing Sandbox Task Finished")
