# -*- coding: utf-8 -*-
import logging
import traceback
import datetime
from sandbox.projects.inventori.common.resources import InventoriAmmunitionFile
import sandbox.sandboxsdk.environments as sandbox_environments
from sandbox.projects.tank.ShootViaTankapi import ShootViaTankapi
import sandbox.sdk2 as sdk2
from sandbox.common.types import task as ctt

LOGGER_NAME = 'inventori load testing'
logger = logging.getLogger(LOGGER_NAME)

query_get_ammunition = '''
use arnold;
pragma yt.InferSchema;
pragma yt.TmpFolder="//home/inventori/tmp";
pragma AnsiInForEmptyOrNullableItemsCollections;

-- скрипт ручного получения патронов для обстрела
$parsed_inventori_runtime_logs = "//home/inventori/logs/daily_parsed_deploy-logs";
-- здесь заносим даты для выкавыривания патронов
$start_date = '{start_date}';
$end_date = '{end_date}';

-- здесь заносим список апи, которые хотим получить
-- "api/general_campaigns_prediction_by_cid", "api/bulk_general_campaigns_prediction", ""
$apis = AsList({apis_list});

$script = @@
from datetime import datetime
import json
import math

def get_dates_to_process(parsed_runtime_paths, start_date, end_date):
    parsed_runtime_dates = [path.decode("utf-8").split("/")[-1] for path in parsed_runtime_paths]

    filtered = sorted([dt for dt in set(parsed_runtime_dates) if dt >= start_date.decode("utf-8") and dt <= end_date.decode("utf-8")])

    return filtered

@@;

$get_dates_to_process = Python::get_dates_to_process(Callable<(List<String>?, String?, String?) -> List<String>?>, $script);

$parsed_runtime_paths = select AGGREGATE_LIST(Path) from FOLDER($parsed_inventori_runtime_logs);

EVALUATE FOR $date in $get_dates_to_process($parsed_runtime_paths, $start_date, $end_date) do begin

    $source_tab = $parsed_inventori_runtime_logs || "/" || $date;

    select
        cast(LENGTH(message_body) as string)||' '||uri||"\n"||message_body as content
        , `timestamp`
    from $source_tab
    where uri in $apis
    and stage = '{stage}'
    and `message_type` = "request"
    LIMIT {rows_limitation}
end do;
'''

DEFAULT_TANK_CONFIG = """
phantom:
  address: inventori-server.2efj3snkc3xpvqyo.iva.yp-c.yandex.net:80
  load_profile: {load_type: rps, schedule: 'line(1,10,200s) const(0, 10s) const(10, 10s)'}
  ssl: true
  writelog: proto_warning
  ammo_type: uripost
uploader:
  operator: inventori
  task: INVENTORI-4018
  job_name: 'Heavy Bullets'
  job_dsc: 'started from Sandbox'
  meta: {use_tank: tank.2efj3snkc3xpvqyo.iva.yp-c.yandex.net:8083}
"""

DEFAULT_TANK = "tank.2efj3snkc3xpvqyo.iva.yp-c.yandex.net:8083"


class InventoriLoadTestingTask(sdk2.Task):
    class Requirements(sdk2.Requirements):
        environments = [
            sandbox_environments.PipEnvironment("yql"),
            sandbox_environments.PipEnvironment("yandex-yt"),
            sandbox_environments.PipEnvironment("yandex-yt-yson-bindings-skynet")
        ]
        disk_space = 80 * 1024  # 80 Gb for data (all necessary indices + archive)
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        yt_yql_cluster = 'arnold'

        yt_token_vault_name = sdk2.parameters.String("YT key vault name", required=True,
                                                     default_value='INVENTORI_YT_TOKEN')
        yt_token_vault_user = sdk2.parameters.String("YT key vault user", required=True, default_value='INVENTORI')

        yql_token_vault_name = sdk2.parameters.String("YQL key vault name", required=True,
                                                      default_value='INVENTORI_YQL_TOKEN')
        yql_token_vault_user = sdk2.parameters.String("YQL key vault user", required=True, default_value='INVENTORI')

        start_date = sdk2.parameters.String("Start Date", required=True, default_value=(
            datetime.datetime.now() - datetime.timedelta(days=5)).strftime("%Y-%m-%d"))
        end_date = sdk2.parameters.String("End Date", required=True,
                                          default_value=datetime.datetime.now().strftime("%Y-%m-%d"))

        stage = sdk2.parameters.String("Stage to form bullets", required=True, default="inventori-prod")
        headers = sdk2.parameters.Dict('headers',
                                       description='Add your values to header or keep default',
                                       default={'Connection': 'Close',
                                                'User-Agent': 'Tank',
                                                'Content-Type': 'application/json'
                                                })

        with sdk2.parameters.RadioGroup("Server type") as server_type:
            server_type.values['media'] = server_type.Value('media', default=True)
            server_type.values['performance'] = server_type.Value('performance')

        MEDIA_APIs = [
            "'/api/budget_performance'",
            "'/api/bulk_campaign_validation'",
            "'/api/bulk_campaigns_prediction'",
            "'/api/bulk_general_campaigns_prediction'",
            "'/api/campaign_validation'",
            "'/api/general_campaign_charts'",
            "'/api/general_campaign_performance'",
            "'/api/general_campaigns_prediction_by_cid'",
            "'/api/group_reach_prediction'",
            "'/api/traffic_light_prediction'",
        ]
        PERFORMANCE_APIs = [
            "'/api/get_events'"
        ]
        with server_type.value['media']:
            with sdk2.parameters.CheckGroup('APIs check list', required=True, do_not_copy=True) as apis_media:
                for med_api in MEDIA_APIs:
                    apis_media.values[med_api] = apis_media.Value(med_api)

        with server_type.value['performance']:
            with sdk2.parameters.CheckGroup('APIs check list', required=True, do_not_copy=True) as apis_performance:
                for perf_api in PERFORMANCE_APIs:
                    apis_performance.values[perf_api] = apis_performance.Value(perf_api)

        include_apis = sdk2.parameters.String("Your custom APIs. Please check that inventori include it")
        rows_limitation = sdk2.parameters.Integer('Bullets(rows) limitation PER DAY. ',
                                                  description='Total bullets number will N_days * rows_limitation',
                                                  required=True, default_value=int(1e4))

        config_content = sdk2.parameters.String('Tank config',
                                                multiline=True,
                                                default_value=DEFAULT_TANK_CONFIG,
                                                required=True
                                                )
        tank = sdk2.parameters.String('Tank address',
                                      required=True,
                                      default_value=DEFAULT_TANK
                                      )

    def fire(self, desc, ammo_resource, config_content, tanks):
        subtask_shoot = ShootViaTankapi(
            self,
            description=desc,
            ammo_source='resource',
            ammo_resource=ammo_resource,
            config_source='file',
            config_content=config_content,
            tanks=tanks,
        ).enqueue()
        logger.info('Subtask with shooting is started')
        raise sdk2.WaitTask([subtask_shoot.id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True,
                            timeout=14400)

    def on_execute(self):
        from yql.api.v1.client import YqlClient
        from yql.client.operation import YqlOperationShareIdRequest

        if not self.Parameters.include_apis and not self.Parameters.apis_media \
            and not self.Parameters.apis_performance:
            raise ValueError("You forget to specify apis parameters")


        logger.info("Inventori Load Testing Sandbox Task Started...")
        yql_proxy = self.Parameters.yt_yql_cluster
        yql_token = sdk2.Vault.data(self.Parameters.yql_token_vault_user,
                                    self.Parameters.yql_token_vault_name)
        start_date = self.Parameters.start_date
        end_date = self.Parameters.end_date
        stage = self.Parameters.stage
        include_apis = str(self.Parameters.include_apis)
        checked_apis = ','.join(self.Parameters.apis_media or self.Parameters.apis_performance)
        all_apis = checked_apis + ',' + include_apis
        yql_client = YqlClient(db=yql_proxy, token=yql_token)
        tank = self.Parameters.tank
        rows_limitation = self.Parameters.rows_limitation

        with self.memoize_stage.create_children:

            def operation_callback(operation):
                share_request = YqlOperationShareIdRequest(operation.operation_id)
                share_request.run()
                share_url = 'https://yql.yandex-team.ru/Operations/{id}'.format(id=share_request.json)
                logger.info('--- YQL shared link on cluster (%s): %s', 'arnold', share_url)

            try:
                prepaired_query = query_get_ammunition.format(start_date=start_date,
                                                              end_date=end_date,
                                                              apis_list=all_apis,
                                                              stage=stage,
                                                              rows_limitation=rows_limitation)
                logger.info("YQL request started")
                logger.info("YQL QUERY:[{query_str}]".format(query_str=prepaired_query))
                start_time = datetime.datetime.now()
                request = yql_client.query(query=prepaired_query, syntax_version=1)
                request.run(pre_start_callback=operation_callback)
                results = request.get_results()

                duration = datetime.datetime.now() - start_time

                if not results.is_success:
                    request.abort()
                    if results.errors:
                        for error in results.errors:
                            logger.error(' -- ERROR: ' + error.format_issue())
                else:
                    logger.info("YQL request finished successfully, duration={duration}".format(
                        duration=duration.total_seconds()))

                resource = InventoriAmmunitionFile(self, "Inventori Ammunition File For Tank Poof Poof", "ammunition")
                ammunition_data = sdk2.ResourceData(resource)

                ammunition_file_path = str(ammunition_data.path)

                headers = '\n'.join(
                    ['[{}: {}]'.format(header, value) for header, value in self.Parameters.headers.items()]
                ) if self.Parameters.headers else ''

                with open(ammunition_file_path, 'w') as ammunition_file:
                    if headers:
                        ammunition_file.write(headers + '\n')
                    for bullet in results.table.get_iterator():
                        if bullet[0]:
                            ammunition_file.write(bullet[0] + '\n')
                logging.info('File %s is filled with ammunition', ammunition_file_path)
                ammunition_data.ready()

                self.set_info("You can find link to lunapark on shooting task. Please see Child task")

                self.fire("Inventori Tank",
                          ammo_resource=resource,
                          config_content=self.Parameters.config_content,
                          tanks=[tank]
                          )
            except Exception as e:
                logger.error("Inventori Load Testing Sandbox Task Failed %s", e)
                logger.error(traceback.format_exc())
            finally:
                logger.info("Inventori Load Testing Sandbox Task Finished")
