# -*- coding: utf-8 -*-

import logging
import sandbox.projects.inventori.common.yql_utils as yql_utils
import sandbox.projects.inventori.common.yt_utils as yt_utils
import sandbox.sandboxsdk.environments as sdk_environments
import sandbox.sdk2 as sdk2
from jinja2 import Template

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

# Platform runtime logs
# https://wiki.yandex-team.ru/qloud/doc/logs/#yt
#
PLATFORM_RUNTIME_DAILY_LOGS = "//home/logfeller/logs/qloud-runtime-log/1d"

# Platform balancers logs
# https://wiki.yandex-team.ru/qloud/doc/logs/#balanceryt
#
PLATFORM_BALANCER_DAILY_LOGS_FOLDER = "//home/logfeller/logs/qloud-router-log/1d"

INVENTORI_RUNTIME_DAILY_LOGS = "//home/inventory/daily_runtime_logs"
INVENTORI_BALANCER_DAILY_LOGS = "//home/inventory/daily_router_logs"


class InventoriLogsTransfer(sdk2.Task):
    class Requirements(sdk2.Requirements):
        environments = [
            sdk_environments.PipEnvironment("yandex-yt"),
            sdk_environments.PipEnvironment("yql")
        ]

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group("Environment parameters") as environment_parameters:
            yt_token_vault_name = sdk2.parameters.String('YT_TOKEN vault name',
                                                         default='INVENTORI_YT_TOKEN',
                                                         required=True)

            yt_token_vault_user = sdk2.parameters.String('YT_TOKEN vault user',
                                                         default='INVENTORI',
                                                         required=True)

            yt_proxy = sdk2.parameters.String('YT_PROXY',
                                              default='hahn',
                                              required=True)

            yql_token_vault_name = sdk2.parameters.String('YQL_TOKEN vault name',
                                                          default='INVENTORI_YQL_TOKEN',
                                                          required=True)

            yql_token_vault_user = sdk2.parameters.String('YQL_TOKEN vault user',
                                                          default='INVENTORI',
                                                          required=True)

            yql_proxy = sdk2.parameters.String('YQL_PROXY',
                                               default='hahn',
                                               required=True)

    def get_processing_dates(self, src_tables, dst_tables):
        return src_tables if not dst_tables else list(set(src_tables) - set(dst_tables))

    def on_execute(self):

        from yt.wrapper import YtClient
        from yql.api.v1.client import YqlClient

        yt_token = sdk2.Vault.data(self.Parameters.yt_token_vault_user, self.Parameters.yt_token_vault_name)
        yt_client = YtClient(proxy=self.Parameters.yt_proxy, token=yt_token)

        yql_token = sdk2.Vault.data(self.Parameters.yql_token_vault_user, self.Parameters.yql_token_vault_name)
        yql_client = YqlClient(db=self.Parameters.yql_proxy, token=yql_token)

        #
        # making general mapping
        #

        src_to_dst_mapping = {}

        for src_folder, dst_folder in {
            PLATFORM_RUNTIME_DAILY_LOGS: INVENTORI_RUNTIME_DAILY_LOGS,
            PLATFORM_BALANCER_DAILY_LOGS_FOLDER: INVENTORI_BALANCER_DAILY_LOGS
        }.items():
            yt_utils.create_folder_if_not_exist(yt_client, dst_folder)

            logger.info("defining tables to transfer in %s" % src_folder)

            dates_to_process = self.get_processing_dates(
                yt_client.list(src_folder),
                yt_client.list(dst_folder)
            )

            for date in dates_to_process:
                src_table = src_folder + "/" + date
                dst_table = dst_folder + "/" + date
                src_to_dst_mapping[src_table] = dst_table

        logger.info("processing logs: %s" % ', '.join(
            "%s->%s" % (src, dst) for (src, dst) in src_to_dst_mapping.iteritems()))

        #
        # transfer mapping processing
        #

        query = Template(
            '''
                PRAGMA yt.InferSchema;

                {% for src_table, dst_table in src_to_dst_mapping.items() %}
                    INSERT INTO `{{dst_table}}`
                    SELECT
                        *
                    FROM `{{src_table}}`
                    where qloud_application ilike '%mediareach%'
                    ;
                {% endfor %}
        ''').render(
            src_to_dst_mapping=src_to_dst_mapping
        )
        yql_utils.run_yql_command(
            yql=yql_client,
            command_name="--- TRANSFERRING LOGS",
            query=query
        )
