# -*- coding: utf-8 -*-

import datetime
import dateutil.relativedelta as relativedelta
import logging
import sandbox.common.types.notification as ctn
import sandbox.projects.inventori.common.parameters as parameters
from sandbox.projects.inventori.common.resources import ROBOT_INVENTORI_SECRET
import sandbox.sandboxsdk.environments as sdk_environments
import sandbox.sdk2 as sdk2

import consts

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)


class InventoriRemoveOldFoldersAndTables(sdk2.Task):
    """Inventori remove old folders and tables"""

    class Requirements(sdk2.Requirements):
        environments = [
            sdk_environments.PipEnvironment("yandex-yt")
        ]

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group("Environment parameters") as environment_parameters:
            environment_type = parameters.EnvironmentParameters.environment_type()

            yt_project_dir = parameters.EnvironmentParameters.yt_project_dir()

        with sdk2.parameters.Group("Task parameters") as task_parameters:
            paths_for_cleaning = sdk2.parameters.List('Relative paths that needs to be cleared',
                                                      sdk2.parameters.String,
                                                      default=['daily_bases', 'daily_cooked_dsp_log_sample',
                                                               'daily_history_traffic', 'daily_prebases'],
                                                      required=True)

            config_table = sdk2.parameters.String('Relative path to table with config',
                                                  default='common/path_days_of_storage',
                                                  required=True)

            yt_token = sdk2.parameters.YavSecretWithKey(
                'YT token',
                default='#'.join((ROBOT_INVENTORI_SECRET, 'yt_token')),
                required=True,
            )

            yt_proxy = sdk2.parameters.String('YT_PROXY',
                                              default='hahn',
                                              required=True)

    def send_invalid_items(self, current_folder, invalid_items):
        if len(invalid_items) != 0:
            self.server.notification(
                subject=consts.INVALID_ITEMS_EMAIL_SUBJECT.format(folder_name=current_folder),
                recipients=consts.INVALID_ITEMS_EMAIL_RECIPIENTS,
                transport=ctn.Transport.EMAIL,
                body=consts.INVALID_ITEMS_EMAIL_BODY.format(folder_name=current_folder, list_of_items=invalid_items)
            )

    def get_dict_from_rows(self, config_table_rows):
        dict_config = {}
        for row in config_table_rows:
            dict_config[row["path"]] = row["days_of_storage"]
        return dict_config

    def validate_items(self, items):
        valid_items = []
        invalid_items = []
        for item in items:
            try:
                datetime.datetime.strptime(item, consts.DATE_FORMAT)
                valid_items.append(item)
            except ValueError:
                invalid_items.append(item)
        return valid_items, invalid_items

    def on_execute(self):
        parameters.init_yt_params(self)

        import yt.wrapper as yt
        import yt.wrapper.ypath as ypath

        yt_token = self.Parameters.yt_token.data()[self.Parameters.yt_token.default_key]
        yt_client = yt.YtClient(proxy=self.Parameters.yt_proxy, token=yt_token)
        path_to_config_table = ypath.ypath_join(self.yt_project_dir, self.Parameters.config_table)

        config_table_rows = yt_client.read_table(path_to_config_table, format=yt.JsonFormat())
        dict_config = self.get_dict_from_rows(config_table_rows)

        for path_for_cleaning in self.Parameters.paths_for_cleaning:

            days_to_clean = dict_config.get(path_for_cleaning)
            if days_to_clean is None:
                continue

            current_folder = ypath.ypath_join(self.yt_project_dir, path_for_cleaning)

            logger.info('Start handle %s folder, days to clean - %i', current_folder, days_to_clean)

            all_items = yt_client.list(current_folder)
            valid_items, invalid_items = self.validate_items(all_items)

            max_date = datetime.datetime.strptime(max(valid_items), consts.DATE_FORMAT)
            min_item = (max_date - relativedelta.relativedelta(
                days=days_to_clean)).strftime(consts.DATE_FORMAT)

            logger.info('%s folder, max date - %s, min date - %s', max_date.strftime(consts.DATE_FORMAT), min_item)

            for item in valid_items:
                if item < min_item:
                    item_full_path = ypath.ypath_join(current_folder, item)
                    yt_client.remove(item_full_path, recursive=True)
                    logger.info('Item %s removed', item_full_path)

            logger.info('Finish handle %s folder', current_folder)

            self.send_invalid_items(current_folder, invalid_items)
