# -*- coding: utf-8 -*-
import datetime
import logging
import textwrap

import sandbox.sdk2 as sdk2
from sandbox.common.types.task import Status as TaskStatus
from yql.api.v1.client import YqlClient
from yql.client.operation import YqlOperationShareIdRequest
from yt.wrapper import YtClient

from yabs.autobudget.pylibs.solomon import Solomon
from sandbox.projects.inventori.common.resources import ROBOT_INVENTORI_SECRET

logger = logging.getLogger('InventoriServerLogMonitoring')


TASK_NAME = 'server_log_monitoring'
STATUS_START = 0
STATUS_OK = 1
STATUS_FAIL = -1


class Text(sdk2.parameters.String):
    multiline = True


class InventoriServerLogMonitoring(sdk2.Task):

    class Requirements(sdk2.Requirements):
        disk_space = 500  # 500 MiB

        # Requirements for multislot agents
        cores = 1  # < 16
        ram = 512  # < 64 GiB

        class Caches(sdk2.Requirements.Caches):
            pass  # Do not use any shared caches (required for running on multislot agent)

    class Parameters(sdk2.Parameters):

        yt_yql_proxy = sdk2.parameters.String('YT/YQL cluster', default='arnold', required=True)
        yt_token = sdk2.parameters.YavSecretWithKey(
            'YT token',
            default='#'.join((ROBOT_INVENTORI_SECRET, 'yt_token')),
            required=True,
        )
        yql_token = sdk2.parameters.YavSecretWithKey(
            'YT token',
            default='#'.join((ROBOT_INVENTORI_SECRET, 'yql_token')),
            required=True,
        )

        log_dir_path = sdk2.parameters.String(
            'Path to log dir',
            default='//logs/inventori-awacs-l7-access-log-test/stream/5min',
            required=True,
        )
        log_start_table = sdk2.parameters.String(
            'Log first table to process from',
        )

        max_batch_size = sdk2.parameters.Integer('Max batch size', default=12)

        solomon_token = sdk2.parameters.YavSecretWithKey(
            'Solomon token',
            default='#'.join((ROBOT_INVENTORI_SECRET, 'solomon_token')),
        )
        solomon_project = sdk2.parameters.String('Solomon project', default='inventori')
        solomon_cluster = sdk2.parameters.String('Solomon cluster', default='test')
        solomon_service = sdk2.parameters.String('Solomon service', default='error_monitoring')

        query = Text(
            'YQL query',
            default=textwrap.dedent('''\
                PRAGMA dq.AnalyzeQuery = '1';
                PRAGMA AnsiInForEmptyOrNullableItemsCollections;

                SELECT
                    `timestamp` AS time,
                    status AS status_code,
                    COUNT(*) AS cnt,
                FROM
                    {select_from}
                WHERE
                    status IN (500, 504)
                GROUP BY
                    `timestamp`, status
                ;
            '''),
            required=True,
        )

    def process_server_log(self):
        yt_client = YtClient(
            proxy=self.Parameters.yt_yql_proxy,
            token=self.Parameters.yt_token.data()[self.Parameters.yt_token.default_key],
        )

        yql_client = YqlClient(
            db=self.Parameters.yt_yql_proxy,
            token=self.Parameters.yql_token.data()[self.Parameters.yql_token.default_key],
        )

        solomon_client = Solomon(
            oauth_token=self.Parameters.solomon_token.data()[self.Parameters.solomon_token.default_key],
            project=self.Parameters.solomon_project,
            cluster=self.Parameters.solomon_cluster,
            service=self.Parameters.solomon_service,
            logger=logger,
        )

        prev_last_processed_table = ''
        if self.scheduler and self.scheduler > 0:
            last_succeeded_task = sdk2.Task.find(scheduler=self.scheduler, status=TaskStatus.SUCCESS, order='-id', limit=1).first()
            if last_succeeded_task:
                prev_last_processed_table = getattr(last_succeeded_task.Context, 'last_processed_table', '')

        tables = list(sorted(
            x for x in yt_client.list(self.Parameters.log_dir_path, absolute=True)
            if x > prev_last_processed_table and x >= self.Parameters.log_start_table
        ))[:self.Parameters.max_batch_size]

        logger.info('Lst tables: %s', tables)

        if tables:
            self.Context.last_processed_table = tables[-1]
        else:
            self.Context.last_processed_table = prev_last_processed_table
            logger.info('There is no new logs')
            return

        query = self.Parameters.query.format(
            select_from="RANGE(`{}`, '{}', '{}')".format(
                self.Parameters.log_dir_path,
                tables[0][len(self.Parameters.log_dir_path):].lstrip('/'),
                tables[-1][len(self.Parameters.log_dir_path):].lstrip('/'),
            ),
        )
        logger.info('Query:\n%s', query)

        def operation_callback(operation):
            share_request = YqlOperationShareIdRequest(operation.operation_id)
            share_request.run()
            share_url = 'https://yql.yandex-team.ru/Operations/{id}'.format(id=share_request.json)
            logger.info('-- YQL shared link on cluster (%s): %s', self.Parameters.yt_yql_proxy, share_url)

        request = yql_client.query(query)
        request.run(pre_start_callback=operation_callback)
        results = request.get_results()

        if not results.is_success:
            request.abort()

            logger.error('request failed: status=%s', results.status)
            logger.error('request json: %s', results.json)
            if results.errors:
                for error in results.errors:
                    logger.error('returned error: %s', error)
            raise Exception('YQL request failed')

        tables = list(results)
        result_table = tables[0]
        sensors_list = []
        for row in result_table.get_iterator():
            row_dict = dict(zip(result_table.column_names, row))
            sensors_list.append({
                'ts': row_dict['time'],
                'labels': {
                    'metric': 'count',
                    'status_code': str(row_dict['status_code']),
                },
                'value': row_dict['cnt'],
            })
        if sensors_list:
            solomon_client.send(sensors_list)

    def on_execute(self):
        logger.info('INVENTORI_SERVER_LOG_MONITORING')

        now = datetime.datetime.now()
        now_ts = int(now.timestamp())

        scheduler_solomon_client = Solomon(
            oauth_token=self.Parameters.solomon_token.data()[self.Parameters.solomon_token.default_key],
            project=self.Parameters.solomon_project,
            cluster=self.Parameters.solomon_cluster,
            service='scheduler',
            logger=logger,
        )

        def send_status(status):
            scheduler_solomon_client.send([{
                'ts': now_ts,
                'labels': {
                    'metric': 'status',
                    'task_name': TASK_NAME
                },
                'value': status,
            }])

        send_status(STATUS_START)

        try:
            self.process_server_log()
        except Exception:
            try:
                send_status(STATUS_FAIL)
            except Exception:
                logger.exception('Failed to send scheduling status to solomon')
            raise
        else:
            send_status(STATUS_OK)
