import copy
import sys

import six
from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types import resource as ctr
from sandbox.common.types import task as ctt


MISSING = object()


if sys.version_info < (3, 4):
    _original_max = max

    def max(iterable, key=MISSING, default=MISSING):
        kwargs = {}
        if key is not MISSING:
            kwargs.update(key=key)
        try:
            return _original_max(iterable, **kwargs)
        except ValueError:
            if default is MISSING:
                raise
            return default


class LastBinaryTaskRelease(object):
    """
    Mixin class to automatically find last resource with binary task executor according to actual parameter value
    and setup proper task requirement. Overrides on_save and on_execute methods.
    Search of the resource is performed by "task_type" attribute which should be explicitly set and
    "released" attribute which is set automatically when resource is released (see
    https://wiki.yandex-team.ru/sandbox/releases/). To configure resource search,
    you need to override the binary_executor_query_iter generator method,
    which should yield dictionaries with valid arguments
    for sdk2.Resource.find().

    Parameters of the task being mixed should either inherit LastBinaryReleaseParameters or include it.

    One can use module methods instead of inheriting mixin class.
    """

    def binary_executor_query_iter(self):
        yield {
            'attrs': {
                'task_type': self.type.name,  # noqa
                'released': self.Parameters.binary_executor_release_type,  # noqa
            },
            'state': [ctr.State.READY],
        }

    def on_save(self):
        setup_requirement(self)  # noqa

    def on_execute(self):
        validate_resource_executor(self)  # noqa


TESTING_OR_UPPER = 'testing_or_upper'
CUSTOM = 'custom'


BINARY_TYPES = [
    ctt.ReleaseStatus.STABLE,
    TESTING_OR_UPPER,
    CUSTOM,
]


def binary_release_parameters_list(
    stable=False,
    testing_or_upper=False,
    custom=False
):
    """
    Inventori analogue of: https://a.yandex-team.ru/arc_vcs/sandbox/projects/common/binary_task/__init__.py?rev
    =r9018843#L79
    """

    kws = locals().copy()
    release = None

    for _release, value in six.iteritems(kws):
        if value:
            if release:
                raise ValueError('Only one of {} can be True'.format(BINARY_TYPES))
            else:
                release = _release
    if not release:
        raise ValueError('Release must be one of {}'.format(BINARY_TYPES))

    class _LastBinaryReleaseParameters(sdk2.Parameters):
        with sdk2.parameters.Group('Binary task executor') as binary_executor_release_type_group:
            binary_executor_release_type = sdk2.parameters.String(
                'Release type of binary executor resource',
                required=True,
                default=release,
                choices=[(t, t) for t in BINARY_TYPES], ui=sdk2.parameters.String.UI('select')
            )

    return _LastBinaryReleaseParameters()


def query_the_most_fresh_of(query_iter):
    return max(
        filter(
            None,
            (
                sdk2.service_resources.SandboxTasksBinary.find(**query).order(
                    -sdk2.Resource.created
                ).first()
                for query in query_iter
            )
        ),
        key=lambda r: r.created,
        default=None,
    )


def setup_requirement(task):
    if task.Parameters.binary_executor_release_type == CUSTOM:
        return

    if task.Parameters.binary_executor_release_type == TESTING_OR_UPPER:
        task.Requirements.tasks_resource = query_the_most_fresh_of(
            dict(copy.deepcopy(query), attrs=dict(copy.deepcopy(query['attrs']), released=str(env)))
            for query in task.binary_executor_query_iter()
            for env in (ctt.ReleaseStatus.TESTING, ctt.ReleaseStatus.STABLE)
        )
    else:
        task.Requirements.tasks_resource = query_the_most_fresh_of(task.binary_executor_query_iter())

    validate_resource_executor(task)


def validate_resource_executor(task):
    if (
        task.Parameters.binary_executor_release_type != CUSTOM
        and task.Requirements.tasks_resource is None
    ):
        raise errors.TaskFailure(
            "Can't find any SandboxTasksBinary resource for task type {} with attribute released=={}".format(
                task.type.name, task.Parameters.binary_executor_release_type
            )
        )
