import logging
import os
import signal
import time

import sandbox.sdk2 as sdk2

from sandbox.projects.rope import parameters
from sandbox.projects.rope.RunTaskTemplate import BinaryResourceExecutor

SECONDS_WAITING_FOR_YQL_LINK_FROM_BINARY = 7

MESSAGE_FOR_SHARED_LINK = 'YQL shared link on cluster'


class InventoriBinaryResourceExecutor(BinaryResourceExecutor):
    def run(self):
        task_executable_data = sdk2.ResourceData(self.task_executable_resource)
        task_executable_data_path = str(task_executable_data.path)
        if isinstance(self.task_params, dict):
            task_env_ars = self.task_params
        elif isinstance(self.task_params, parameters.TaskParams):
            task_env_ars = self.task_params.to_env_args()
        else:
            raise ValueError('task_param has unsupported type {}'.format(type(self.task_params)))

        with sdk2.helpers.ProcessLog(self.sdk2_sb_obj, logger=self.logger) as pl:
            process = sdk2.helpers.subprocess.Popen(
                [task_executable_data_path],
                stdout=pl.stdout,
                stderr=pl.stderr,
                env=task_env_ars,
            )

            time.sleep(SECONDS_WAITING_FOR_YQL_LINK_FROM_BINARY)

            yql_shared_link = None

            with open(str(self.sdk2_sb_obj.log_path('common.log'))) as current_log:
                for line in current_log:
                    if MESSAGE_FOR_SHARED_LINK in line:
                        yql_shared_link = line
                        break

            if yql_shared_link:
                self.sdk2_sb_obj.set_info(
                    "Shared link: <a href={0}> {0} </a>".format(yql_shared_link.split()[-1]),
                    do_escape=False,
                )

            process.wait()
            return_code = process.poll()
            if return_code is not None and return_code != 0:
                self.logger.info('return_code is {}'.format(return_code))
                raise ValueError('Problem in binary task, pls check logs')

    @classmethod
    def on_stop(cls, sdk2_sb_obj, logger=None):
        logger = logger or logging
        for process in sdk2.helpers.ProcessRegistry:
            try:
                logger.info('Sending SIGINT to binary %s', process.pid)
                os.kill(process.pid, signal.SIGINT)
                time.sleep(3)

                logging.info('Sending SIGTERM to binary')
                try:
                    os.kill(process.pid, signal.SIGTERM)
                except ProcessLookupError:
                    logging.info('Product binary probably have been stopped successfully')
                    continue

                time.sleep(5)
                logging.info('Sending SIGKILL to binary')

                try:
                    os.kill(process.pid, signal.SIGKILL)
                except ProcessLookupError:
                    logging.info('Product binary probably have been stopped successfully')

            except OSError:
                logging.info('problems while killing binaries')

