# -*- coding: utf-8 -*-
"""
This module describe all resources for inventori infrastructure

Also some of them could be registered as so called inventori tasks
with `register_binary_inventori_task` decorator.
Then you can build such resource with common BUILD_INVENTORI_TASKS sandbox task
by selecting proper checkbox, and run them with common RUN_INVENTORI_TASKS
by selecting proper radiobutton.

Such resource should consists of single executable binary.
You should provide `arcadia_build_path` argument in form:
<path_to_build_directory>/<bin_name>

It also helpful to provide
* `task_name` - simple and unique task name in snake_case
* `task_executable_resource_description` - resource description

Argument for this task could be provided with inner class
named `InventoriTaskParams` and inherited from `parameters.TaskParamAbstract`.
Inner attributes of this class that are instances of `parameters.StrParam`
will generate proper input fields in sandbox RUN_INVENTORI_TASKS task ui.
This arguments in a form of environment variables will be passed to binary executable.
You could also parse them with `from_cli_args_or_env` fabric class method:

Example:
    @register_binary_inventori_task
    class TestTaskBinaries(sdk2.Resource):
        ...
        arcadia_build_path = 'inventori/test_task/bin/test_task'
        task_name = 'test_task'
        task_executable_resource_description = 'Inventori test task executable binaries'

        class InventoriTaskParams(InventoriTaskParams):
            str_arg = parameters.StrParam(
                'Some string argument',
                required=True,
            )
            int_arg = parameters.StrParam(
                'Some int argument',
                default=13,
                sdk2_param_class=sdk2.parameters.Integer
            )

    params = TestTaskBinaries.InventoriTaskParams.from_cli_args_or_env(env={'STR_ARG': 'hello'})
    assert params.str_arg == 'hello'
    assert params.int_arg == 13
"""
import calendar
import copy
import importlib
import logging
import os
import re
import time
from datetime import datetime
from enum import Enum

from sandbox import sdk2

from sandbox.projects.inventori.common.binary_task_executor import InventoriBinaryResourceExecutor
from sandbox.projects.rope import parameters
from sandbox.projects.rope.RunTaskTemplate import BinaryResourceTaskInitializer
from sandbox.projects.rope.RunTaskTemplate import FunctionTaskInitializer

ROBOT_INVENTORI_SECRET = 'sec-01eb89vakhn60j7ynwmr16a9ne'

INVENTORI_RELEASERS = [
    'stason',
    'kacher',
    'd-frolov',
    'k1noo',
    'akimov-victor',
    'taktarov',
    'nikitakogut',
    'robot-inventori',
    'INVENTORI',
]
GENERAL_TTL = 30  # number of days

INVENTORI_TASK_BINARIES_RESOURCES = []
INVENTORI_TASK_INITIALIZERS = []
INVENTORI_TASK_NAMES = set()

INVENTORI_HOME = '//home/inventori'


def create_inventori_default_path_getter(relative_path=None, require=None, home=INVENTORI_HOME):
    """
        Create default path getter relative from //home/inventori/[prod|test]

        :type relative_path: str | (TaskParams) -> str
        :type require: str | list[str]
        :type home: str
        :param home: home dir path
        :rtype parameters.DefaultGetter
    """
    def func(p):
        """
        :type p: TaskParams | (TaskParams) -> str
        :rtype: str
        """
        if p.environment_type == EnvironmentType.PROD:
            env_folder = EnvironmentType.PROD
        else:
            env_folder = EnvironmentType.TEST

        path = os.path.join(home, env_folder)

        if relative_path:
            if callable(relative_path):
                rel_path = relative_path(p)
            else:
                rel_path = relative_path
            assert not rel_path.startswith('/'), '{} is not a relative path'.format(rel_path)
            path = os.path.join(path, rel_path)

        return path

    return parameters.DefaultGetter(func, require=require)


def default_decorator(decorator):
    def wrapper(*args, **kwargs):
        if not kwargs and len(args) == 1:
            (fn,) = args
            return decorator()(fn)
        return decorator(*args, **kwargs)

    return wrapper


@default_decorator
def register_binary_inventori_task(custom_runner=False):
    """
    :param bool custom_runner: if you want to implement custom runner
        this task will be excluded from common task runner
    """

    def decorator(cls):
        inventori_task_params_class = getattr(cls, 'InventoriTaskParams', None)
        task_initializer = BinaryResourceTaskInitializer(
            task_executable_resource_type=cls,
            task_params_class=inventori_task_params_class,
            binary_res_executor_class=InventoriBinaryResourceExecutor,
        )

        if inventori_task_params_class:
            setattr(inventori_task_params_class, '__default_task_name__', task_initializer.task_name)

        task_name = task_initializer.task_name
        if not re.match(r'^[a-z][a-z0-9]*(_[a-z0-9]+)*$', task_name):
            raise ValueError(
                'Wrong task name "{}" - should be in snake_case.'.format(task_name)
            )
        if task_name in INVENTORI_TASK_NAMES:
            raise ValueError('Task with name {} already exists'.format(task_name))
        INVENTORI_TASK_BINARIES_RESOURCES.append(cls)
        if not custom_runner:
            INVENTORI_TASK_INITIALIZERS.append(task_initializer)
        INVENTORI_TASK_NAMES.add(task_initializer.task_name)
        cls.__is_inventori_task__ = True
        return cls

    return decorator


def register_inventori_task(name, path_to_task_class, legacy_key=None):
    def _no_solomon_status_sending_params(tp):
        tp = copy.deepcopy(tp)
        tp.send_status_to_solomon = False
        return tp

    def import_task_class():
        p = path_to_task_class.split('.')
        task_class_name = p[-1]
        module_path = '.'.join(p[:-1])
        return getattr(importlib.import_module(module_path), task_class_name)

    def func(task_params, sdk2_sb_obj):
        # TODO: remove after debugging
        logging.info(
            'task_name={}'
            ' legacy_key={}'
            ' path_to_task_class={}'
            ' task_params={}'
            ' sdk2_sb_obj={}'
            ''.format(
                name,
                legacy_key,
                path_to_task_class,
                task_params,
                sdk2_sb_obj,
            )
        )

        tp = copy.deepcopy(task_params)
        tp.send_status_to_solomon = False

        task_class = import_task_class()(
            task_params=_no_solomon_status_sending_params(task_params),
            logger=sdk2_sb_obj.logger, sdk2_sb_obj=sdk2_sb_obj)
        task_class.run()

    def on_stop(task_params, sdk2_sb_obj):

        import_task_class()(
            task_params=_no_solomon_status_sending_params(task_params),
            logger=sdk2_sb_obj.logger, sdk2_sb_obj=sdk2_sb_obj,
        ).process_termination()

    def decorator(cls):
        setattr(cls, '__default_task_name__', name)
        INVENTORI_TASK_INITIALIZERS.append(
            FunctionTaskInitializer(
                task_key=legacy_key or name,
                name=name,
                prefix_name=name,
                task_params_class=cls,
                func=func,
                on_stop=on_stop,
            ),
        )
        return cls

    return decorator


class RunMode(str, Enum):
    ASYNC = 'async'
    REPLICATION = 'replication'
    SIMPLE = 'simple'


DUPLICATION_MODES = [RunMode.ASYNC, RunMode.REPLICATION]


class EnvironmentType(str, Enum):
    PROD = 'prod'
    TEST = 'test'
    DEV = 'dev'


class InventoriBaseTaskParams(parameters.TaskParams):
    # for test compatible
    yt_proxy = parameters.StrParam(
        'YT cluster',
        default=lambda p: p.yt_yql_cluster,
        skip_in=parameters.ParamSrc.SANDBOX_SDK2_PARAM,
    )

    # for test compatible
    yql_proxy = parameters.StrParam(
        'YQL cluster',
        default=lambda p: p.yt_yql_cluster,
        skip_in=parameters.ParamSrc.SANDBOX_SDK2_PARAM,
    )

    yt_token = parameters.YavSecretParam(
        'YT token',
        default_yav_secret=ROBOT_INVENTORI_SECRET,
        default_yav_secret_key='yt_token',
    )

    yt_pool = parameters.StrParam('YT pool')

    yql_token = parameters.YavSecretParam(
        'YQL token',
        default_yav_secret=ROBOT_INVENTORI_SECRET,
        default_yav_secret_key='yql_token',
    )
    yql_port = parameters.StrParam(
        'YQL port',
        details='if specified local yql server should be used (for test purpose)',
        skip_in=parameters.ParamSrc.SANDBOX_SDK2_PARAM,
    )

    environment_type = parameters.StrParam(
        'Environment type',
        default='test',
        required=True,
        sdk2_param_class=sdk2.parameters.RadioGroup,
        sdk2_param_kwargs=dict(
            choices=[(env_type.value, env_type.value) for env_type in EnvironmentType]
        ),
        details='This variable represent solomon cluster'
    )

    # U should specify task_name in parameters of your task
    task_name = parameters.StrParam(
        'Task name',
        details='Task name for monitoring (selected task type name by default;'
                ' also will be added as scheduler/task tag)',
        default=lambda p: p.__default_task_name__,
    )

    send_status_to_solomon = parameters.BoolParam('Sent status (-1,0,1) to solomon',
                                                  default=True,
                                                  details="Uncheck if don't want write in Solomon"
                                                  )

    exec_time = parameters.DateTimeParam(
        'Execution time',
        details='Current time by default. Rewrite to execute as from the past or from the future',
        default=lambda p: datetime.now(),
    )

    solomon_token = parameters.YavSecretParam(
        'Solomon token',
        default_yav_secret=ROBOT_INVENTORI_SECRET,
        default_yav_secret_key='solomon_token',
        details='If specified, task monitoring will be set automatically',
        env_arg='INVENTORI_SOLOMON_TOKEN',
        depend_on=('send_status_to_solomon', True)
    )

    oauth_token = parameters.YavSecretParam(
        'Oauth',
        default_yav_secret=ROBOT_INVENTORI_SECRET,
        default_yav_secret_key='oauth'
    )

    log_level = parameters.StrParam(
        'Log level',
        default='INFO',
        # choices=['CRITICAL', 'ERROR', 'WARNING', 'INFO', 'DEBUG'],
        skip_in=parameters.ParamSrc.SANDBOX_SDK2_PARAM,
    )

    is_custom_run_mode_for_this_task = parameters.BoolParam(
        'Using custom run_mode',
        default=False,
        details='default for all tasks is "simple"',
        skip_in=parameters.ParamSrc.CLI_OR_ENV,
    )

    run_mode = parameters.StrParam(
        'Way to duplicate output tables',
        required=parameters.ParamSrc.SANDBOX_SDK2_PARAM,
        default=RunMode.SIMPLE.value,
        sdk2_param_class=sdk2.parameters.RadioGroup,
        sdk2_param_kwargs=dict(
            choices=[(run_mode.value, run_mode.value) for run_mode in RunMode]
        ),
        depend_on=('is_custom_run_mode_for_this_task', True)
    )

    yt_yql_cluster = parameters.StrParam(
        'YT/YQL cluster',
        required=parameters.ParamSrc.SANDBOX_SDK2_PARAM,
        default='hahn',
        sdk2_param_class=sdk2.parameters.RadioGroup,
        sdk2_param_kwargs=dict(
            choices=(
                ('hahn', 'hahn'),
                ('arnold', 'arnold'),
            )
        ),
        depend_on=('run_mode', RunMode.SIMPLE.value)
    )

    using_log_dir = parameters.StrParam(
        'Log dir for choose master\\slave cluster.',
        required=False,
        depend_on=('run_mode', RunMode.REPLICATION.value),
    )

    using_tables = parameters.ListParam(
        'Tables for choose master\\slave cluster. '
        'You can also specify attribute based on which to prioritize cluster. '
        'By default /@modification_time will be used.',
        required=False,
        depend_on=('run_mode', RunMode.REPLICATION.value),
    )

    output_tables = parameters.ListParam(
        'Tables You want to copy on another cluster',
        required=False,
        depend_on=('run_mode', RunMode.REPLICATION.value),
    )

    is_waiting_for_tm_tasks = parameters.BoolParam(
        'Waiting for tm tasks',
        default=True,
        depend_on=('run_mode', RunMode.REPLICATION.value),
        details='Yes, if u are going to wait for replication tasks to finish copying'
    )


class InventoriLogSamplerBinaries(sdk2.Resource):
    releasable = True
    executable = True
    releasers = INVENTORI_RELEASERS
    ttl = GENERAL_TTL

    arcadia_build_path = 'inventori/bin/log_sampler/log_sampler'
    task_name = 'log_sampler'
    task_executable_resource_description = 'Inventori log sampler executable binaries'

    @register_inventori_task(
        legacy_key='INVENTORI_LOG_SAMPLER_BINARIES',
        name='log_sampler',
        path_to_task_class='inventori.pylibs.tasks.log_sampler.prepare_log_sample.LogSampler',
    )
    class InventoriTaskParams(InventoriBaseTaskParams):
        default_logs_days_depth = parameters.IntParam(
            'Default logs days depth',
            details='How many days in the past we need check and resample if necessary',
            default=30,
            sdk2_param_class=sdk2.parameters.Integer,
        )
        sample_logs_ttl_in_days = parameters.IntParam(
            'Sampled logs ttl',
            details='New sampled logs ttl in days',
            default=90,
            sdk2_param_class=sdk2.parameters.Integer,
        )
        yt_dir = parameters.StrParam(
            'YT dir',
            details='Base cypress node directory for result tables to store '
                    '(derived from environment_type by default //home/inventori/[prod|test])',
            default=create_inventori_default_path_getter(),
        )
        yt_inventori_params_tab = parameters.StrParam(
            'YT invenroti params tab',
            details='Path to yt table with task parameters '
                    '(derived from yt_dir by default <yt_dir>/common/parameters)',
            default=parameters.DefaultGetter(
                lambda p: '{}/common/parameters'.format(p.yt_dir), require='yt_dir'
            ),
        )
        now = parameters.DateTimeParam(
            'Exec datetime',
            details='datetime to calculate log back from and with ttl starts from it '
                    '(timestamp in seconds or iso time string; current datetime by default)',
            default=lambda p: datetime.utcnow(),
        )


class InventoriAmmunitionFile(sdk2.Resource):
    ttl = 30


class InventoriBackToBackReportFile(sdk2.Resource):
    ttl = 30


class InventoriServerBinaries(sdk2.Resource):
    releasable = True
    executable = True
    releasers = INVENTORI_RELEASERS
    ttl = GENERAL_TTL


class InventoriPerformanceServerBinaries(sdk2.Resource):
    releasable = True
    executable = True
    releasers = INVENTORI_RELEASERS
    ttl = GENERAL_TTL


class InventoriMonitoringBinaries(sdk2.Resource):
    releasable = True
    executable = True
    releasers = INVENTORI_RELEASERS
    ttl = GENERAL_TTL
    arcadia_build_path = 'inventori/bin/monitoring/monitoring'
    task_name = 'monitoring'
    task_executable_resource_description = 'Inventori monitoring executable binaries'

    @register_inventori_task(
        legacy_key='INVENTORI_MONITORING_BINARIES',
        name='monitoring',
        path_to_task_class='inventori.pylibs.tasks.monitoring.monitoring.Monitoring',
    )
    class InventoriTaskParams(InventoriBaseTaskParams):
        sandbox_token = parameters.YavSecretParam(
            'Sandbox token',
            default_yav_secret=ROBOT_INVENTORI_SECRET,
            default_yav_secret_key='sandbox_token',
            env_arg='SANDBOX_TOKEN',
        )

        now_timestamp = parameters.IntParam(
            'Timestamp for freshness sensors',
            details='Will be used as timestamp while send freshness to Solomon (now by default)',
            default=parameters.DefaultGetter(
                lambda p: int(time.mktime(p.exec_time.timetuple())), require='exec_time')
        )

        override_log_dirs_to_monitor = parameters.BoolParam(
            'Override log dirs to monitor',
            default=False,
            skip_in=parameters.ParamSrc.NOT_SKD2_PARAM,
        )

        log_dirs_to_monitor = parameters.ListParam(
            'Log dirs to monitor',
            depend_on=('override_log_dirs_to_monitor', True),
        )


class InventoriDatabase(sdk2.Resource):
    releasable = True
    releasers = INVENTORI_RELEASERS
    any_arch = True
    auto_backup = True
    ttl = GENERAL_TTL


@register_binary_inventori_task(custom_runner=True)
class InventoriDbMakerBinaries(sdk2.Resource):
    releasable = True
    releasers = INVENTORI_RELEASERS
    any_arch = True
    auto_backup = True
    ttl = GENERAL_TTL
    arcadia_build_path = 'inventori/etl/bin/etl'
    task_name = 'db_maker'
    task_executable_resource_description = 'Inventori db maker executable binaries'

    class InventoriTaskParams(InventoriBaseTaskParams):
        force_validation = parameters.BoolParam(
            'Force database validation', default=True
        )
        yt_dir = parameters.StrParam(
            'YT dir',
            details='Base cypress node directory for result tables to store '
                    '(derived from environment_type by default //home/inventori/[prod|test])',
            default=create_inventori_default_path_getter(),
        )
        yt_database_dir = parameters.StrParam(
            'YT database dir',
            details='Path to store database results '
                    '(derived from yt_dir by default <yt_dir>/database)',
            default=parameters.DefaultGetter(
                lambda p: '{}/database'.format(p.yt_dir), require='yt_dir'
            ),
        )
        now = parameters.DateTimeParam(
            'Exec datetime',
            details='datetime to calculate make db for'
                    '(timestamp in seconds or iso time string; current datetime by default)',
            default=lambda p: datetime.now(),
        )
        yt_current_archive_dir = parameters.StrParam(
            'Current archive result dir',
            details='dir to make tar.gz for deploy '
                    '(derived from yt_database_dir and now by default <yt_dir>/database/YYYY-MM-DD)',
            default=parameters.DefaultGetter(
                lambda p: '{}/daily_bases/{}'.format(p.yt_database_dir, p.now.date()),
                ['yt_database_dir', 'now'],
            ),
        )


@register_binary_inventori_task(custom_runner=True)
class InventoriBacktestingDatabase(sdk2.Resource):
    releasable = True
    releasers = INVENTORI_RELEASERS
    any_arch = True
    auto_backup = True
    ttl = GENERAL_TTL
    arcadia_build_path = (
        'inventori/bin/backtesting/make_backtesting_db/make_backtesting_db'
    )
    task_name = 'make_backtesting_db'
    task_executable_resource_description = (
        'inventori backtesting etl executable binaries'
    )

    class InventoriTaskParams(InventoriBaseTaskParams):
        yt_dir = parameters.StrParam(
            'YT dir',
            details='Base cypress node directory for result tables to store '
                    '(derived from environment_type by default //home/inventori/[prod|test])',
            default=create_inventori_default_path_getter(),
        )
        yt_active_campaigns_registry_dir = parameters.StrParam(
            'YT active campaigns registry dir',
            details='Path to yt dir with active campaigns '
                    '(derived from yt_dir by default <yt_dir>/campaigns_data/common)',
            default=parameters.DefaultGetter(
                lambda p: '{}/campaigns_data/common'.format(p.yt_dir), require='yt_dir'
            ),
        )
        yt_backtesting_dir = parameters.StrParam(
            'YT backtesting dir',
            details='Path to store backtesting results '
                    '(derived from yt_dir by default <yt_dir>/backtesting)',
            default=parameters.DefaultGetter(
                lambda p: '{}/backtesting'.format(p.yt_dir), require='yt_dir'
            ),
        )
        now = parameters.DateTimeParam(
            'Exec datetime',
            details='datetime to calculate backtesting delay from it'
                    '(timestamp in seconds or iso time string; current datetime by default)',
            default=lambda p: datetime.now(),
        )


class DailyCommonCampaignsPreparationBinaries(sdk2.Resource):
    releasable = True
    releasers = INVENTORI_RELEASERS
    executable = True
    ttl = GENERAL_TTL
    arcadia_build_path = (
        'inventori/bin/etl/daily_common_campaigns_preparation/daily_common_campaigns_preparation'
    )
    task_name = 'daily_common_campaigns_preparation'
    task_executable_resource_description = (
        'Inventori daily active campaigns preparation executable'
    )

    @register_inventori_task(
        legacy_key='DAILY_COMMON_CAMPAIGNS_PREPARATION_BINARIES',
        name='daily_common_campaigns_preparation',
        path_to_task_class='inventori.pylibs.tasks.etl.daily_common_campaigns_preparation.'
                           'DailyCommonCampaignsPreparation',
    )
    class InventoriTaskParams(InventoriBaseTaskParams):
        tvm_client_id = parameters.StrParam(
            'InventORI TVM client id', default='2009954', required=True
        )
        tvm_direct_intapi_client_id = parameters.StrParam(
            'Direct intapi TVM client id', default='2000390', required=True
        )
        tvm_secret = parameters.YavSecretParam(
            'TVM secret key',
            default_yav_secret=ROBOT_INVENTORI_SECRET,
            default_yav_secret_key='inventori_to_direct_internal_api_tvm_ticket',

        )

        direct_intapi_host = parameters.StrParam(
            'Direct intapi host',
            default='http://intapi.direct.yandex.ru/inventori',
            required=True,
        )
        direct_intapi_chunk_size = parameters.IntParam(
            'Direct intapi request chunk size',
            default=200,
            required=True,
        )
        direct_intapi_workers_count = parameters.IntParam(
            'Direct intapi request workers count',
            default=3,
            required=True,
        )
        direct_intapi_timeout = parameters.IntParam(
            'Direct intapi request timeout (sec)',
            default=60,
            env_arg='DIRECT_INTAPI_REQUEST_TIMEOUT',
            required=True,
        )

        inventori_api_host = parameters.StrParam(
            'InventORI validation API host',
            default='https://inventori.yandex-team.ru/api/bulk_campaign_validation',
            required=True,
        )
        inventori_api_chunk_size = parameters.IntParam(
            'InventORI validation API request chunk size',
            default=20,
            required=True,
        )
        inventori_api_workers_count = parameters.IntParam(
            'InventORI validation API request workers count',
            default=2,
            required=True,
        )
        inventori_api_timeout = parameters.IntParam(
            'InventORI validation API request timeout (sec)',
            default=60,
            required=True,
        )

        force_applying_path = parameters.StrParam(
            'YT-Path to boolean_node for allowance to apply failed validation result',
        )

        is_debug_mode = parameters.BoolParam('Enable debug mode', default=False)

        debug_active_cids_path = parameters.StrParam(
            'Active cids YT table for debug',
            required=False,
            env_arg='ACTIVE_CIDS_TABLE',
            depend_on=('is_debug_mode', True),
        )

        yt_root_path = parameters.StrParam(
            'YT working directory root path',
            default='//home/inventori',
            required=True,
        )

        apply_invalid_campaigns = parameters.BoolParam(
            'Apply invalid campaigns',
            default=True,
        )


class DailyMainPageBookingCampaignsPreparationBinaries(sdk2.Resource):
    releasable = True
    releasers = INVENTORI_RELEASERS
    executable = True
    ttl = GENERAL_TTL
    arcadia_build_path = (
        'inventori/bin/etl/daily_main_page_booking_campaigns_preparation'
        '/daily_main_page_booking_campaigns_preparation'
    )
    task_name = 'daily_main_page_booking_campaigns_preparation'
    task_executable_resource_description = (
        'Inventori daily active campaigns preparation executable'
    )

    @register_inventori_task(
        legacy_key='DAILY_MAIN_PAGE_BOOKING_CAMPAIGNS_PREPARATION_BINARIES',
        name='daily_main_page_bookings_preparation',
        path_to_task_class='inventori.pylibs.tasks.etl'
                           '.daily_main_page_booking_campaigns_preparation'
                           '.DailyMainPageBookingCampaignsPreparation',
    )
    class InventoriTaskParams(InventoriBaseTaskParams):
        tvm_client_id = parameters.StrParam(
            'InventORI TVM client id', default='2009954', required=True
        )
        tvm_direct_intapi_client_id = parameters.StrParam(
            'Direct intapi TVM client id', default='2000390', required=True
        )
        tvm_secret = parameters.YavSecretParam(
            'TVM secret key',
            default_yav_secret=ROBOT_INVENTORI_SECRET,
            default_yav_secret_key='inventori_to_direct_internal_api_tvm_ticket',

        )

        direct_intapi_host = parameters.StrParam(
            'Direct intapi host',
            default='http://intapi.direct.yandex.ru/inventori',
            required=True,
        )
        direct_intapi_chunk_size = parameters.IntParam(
            'Direct intapi request chunk size',
            default=200,
            required=True,
        )
        direct_intapi_workers_count = parameters.IntParam(
            'Direct intapi request workers count',
            default=3,
            required=True,
        )
        direct_intapi_timeout = parameters.IntParam(
            'Direct intapi request timeout (sec)',
            default=60,
            env_arg='DIRECT_INTAPI_REQUEST_TIMEOUT',
            required=True,
        )

        inventori_api_host = parameters.StrParam(
            'InventORI validation API host',
            default='https://inventori.yandex-team.ru/api/bulk_campaign_validation',
            required=True,
        )
        inventori_api_chunk_size = parameters.IntParam(
            'InventORI validation API request chunk size',
            default=20,
            required=True,
        )
        inventori_api_workers_count = parameters.IntParam(
            'InventORI validation API request workers count',
            default=2,
            required=True,
        )
        inventori_api_timeout = parameters.IntParam(
            'InventORI validation API request timeout (sec)',
            default=60,
            required=True,
        )

        force_applying_path = parameters.StrParam(
            'YT-Path to boolean_node for allowance to apply failed validation result',
        )

        is_debug_mode = parameters.BoolParam('Enable debug mode', default=False)

        debug_active_cids_path = parameters.StrParam(
            'Active cids YT table for debug',
            required=False,
            env_arg='ACTIVE_CIDS_TABLE',
            depend_on=('is_debug_mode', True),
        )

        yt_root_path = parameters.StrParam(
            'YT working directory root path',
            default='//home/inventori',
            required=True,
        )

        apply_invalid_campaigns = parameters.BoolParam(
            'Apply invalid campaigns',
            default=True,
        )


class InventoriCampaignPredictionsExportBinaries(sdk2.Resource):
    releasable = True
    releasers = INVENTORI_RELEASERS
    executable = True
    ttl = GENERAL_TTL
    arcadia_build_path = (
        'inventori/bin/etl/campaign_predictions_export/campaign_predictions_export'
    )
    task_name = 'campaign_predictions_export'
    task_executable_resource_description = (
        'Inventori campaign predictions export executable binaries'
    )

    @register_inventori_task(
        legacy_key='INVENTORI_CAMPAIGN_PREDICTIONS_EXPORT_BINARIES',
        name='campaign_predictions_export',
        path_to_task_class='inventori.pylibs.tasks.etl.campaign_predictions_export'
                           '.CampaignPredictionsExportTask',
    )
    class InventoriTaskParams(InventoriBaseTaskParams):
        tvm_client_id = parameters.StrParam(
            'InventORI TVM client id', default='2009954', required=True
        )
        tvm_direct_intapi_client_id = parameters.StrParam(
            'Direct intapi TVM client id', default='2000390', required=True
        )
        tvm_secret = parameters.YavSecretParam(
            'TVM secret key',
            default_yav_secret=ROBOT_INVENTORI_SECRET,
            default_yav_secret_key='inventori_to_direct_internal_api_tvm_ticket',
        )

        inventori_api_host = parameters.StrParam(
            'InventORI prediction API host',
            details='bare host for /api/bulk_general_campaigns_prediction endpoint',
            default='inventori.yandex-team.ru',
            required=True,
        )
        inventori_api_chunk_size = parameters.IntParam(
            'InventORI prediction API request chunk size',
            default=20,
            required=True,
        )
        inventori_api_workers_count = parameters.IntParam(
            'InventORI prediction API request workers count',
            default=2,
            required=True,
        )
        inventori_api_timeout = parameters.IntParam(
            'InventORI prediction API request timeout (sec)',
            default=60,
            required=True,
        )

        is_debug_mode = parameters.BoolParam('Enable debug mode', default=False)

        debug_active_cids_path = parameters.StrParam(
            'Active cids YT table for debug',
            required=True,
            env_arg='ACTIVE_CIDS_TABLE',
            depend_on=('is_debug_mode', True),
        )

        yt_root_path = parameters.StrParam(
            'YT working directory root path',
            default='//home/inventori',
            required=True,
        )

        task_type = parameters.StrParam(
            'Task type',
            details='Type of active campaigns to export',
            sdk2_param_class=sdk2.parameters.RadioGroup,
            sdk2_param_kwargs=dict(choices=(('cpm', 'CPM'),)),
            required=True,
        )


class InventoriProductMonitoringBinaries(sdk2.Resource):
    releasable = True
    releasers = INVENTORI_RELEASERS
    executable = True
    ttl = GENERAL_TTL
    arcadia_build_path = 'inventori/bin/product_monitoring/product_monitoring'
    task_name = 'product_monitoring'
    task_executable_resource_description = (
        'Inventori product monitoring executable binaries'
    )

    @register_inventori_task(
        legacy_key='INVENTORI_PRODUCT_MONITORING_BINARIES',
        name='product_monitoring',
        path_to_task_class='inventori.pylibs.tasks.product_monitoring.ProductMonitoringTask',
    )
    class InventoriTaskParams(InventoriBaseTaskParams):
        operation_id = parameters.StrParam(
            'Operation id to use instead of running new query',
        )
        inventori_api_host = parameters.StrParam(
            'Inventori api host',
            default='https://inventori.yandex-team.ru',
            required=True,
        )


class InventoriMainPageHistoryCollectorBinaries(sdk2.Resource):
    releasable = True
    releasers = INVENTORI_RELEASERS
    executable = True
    ttl = GENERAL_TTL
    arcadia_build_path = 'inventori/bin/timeseries/mainpage/mainpage'
    task_name = 'history_timeseries_mainpage'
    task_executable_resource_description = (
        'Inventori generate mainpage timeseries executable binaries'
    )

    @register_inventori_task(
        legacy_key='INVENTORI_MAIN_PAGE_HISTORY_COLLECTOR_BINARIES',
        name='history_timeseries_mainpage',
        path_to_task_class='inventori.pylibs.tasks.timeseries.make_mainpage_history'
                           '.MainPageHistoryDataCollector',
    )
    class InventoriTaskParams(InventoriBaseTaskParams):
        result_folder = parameters.StrParam(
            'Result folder',
            details='(Default //home/inventori/[prod|test]/timeseries/history_data/mainpage)',
            default=create_inventori_default_path_getter('timeseries/history_data/mainpage'),
        )


class InventoriBannerHistoryCollectorBinaries(sdk2.Resource):
    releasable = True
    releasers = INVENTORI_RELEASERS
    executable = True
    ttl = GENERAL_TTL
    arcadia_build_path = 'inventori/bin/timeseries/banner/banner'
    task_name = 'history_timeseries_banner'
    task_executable_resource_description = (
        'Inventori generate banner timeseries executable binaries'
    )

    @register_inventori_task(
        legacy_key='INVENTORI_BANNER_HISTORY_COLLECTOR_BINARIES',
        name='history_timeseries_banner',
        path_to_task_class='inventori.pylibs.tasks.timeseries.make_banner_history'
                           '.BannerHistoryDataCollector',
    )
    class InventoriTaskParams(InventoriBaseTaskParams):
        result_folder = parameters.StrParam(
            'Result folder',
            details='(Default //home/inventori/[prod|test]/timeseries/history_data/banner)',
            default=create_inventori_default_path_getter('timeseries/history_data/banner'),
        )

        pi_data_folder = parameters.StrParam(
            'Pi data folder',
            details='(Default //home/inventori/[prod|test]/daily_pi_data/chronology)',
            default=create_inventori_default_path_getter('daily_pi_data/chronology'),
        )


class InventoriAutoruHistoryCollectorBinaries(sdk2.Resource):
    releasable = True
    releasers = INVENTORI_RELEASERS
    executable = True
    ttl = GENERAL_TTL
    arcadia_build_path = 'inventori/bin/timeseries/autoru/autoru'
    task_name = 'history_timeseries_autoru'
    task_executable_resource_description = (
        'Inventori generate autoru timeseries executable binaries'
    )

    @register_inventori_task(
        legacy_key='INVENTORI_AUTORU_HISTORY_COLLECTOR_BINARIES',
        name='history_timeseries_autoru',
        path_to_task_class='inventori.pylibs.tasks.timeseries.autoru_history'
                           '.AutoruHistoryDataCollector',
    )
    class InventoriTaskParams(InventoriBaseTaskParams):
        result_folder = parameters.StrParam(
            'Result folder',
            details='(Default //home/inventori/[prod|test]/timeseries/history_data/autoru)',
            default=create_inventori_default_path_getter('timeseries/history_data/autoru'),
        )


class InventoriRealtyHistoryCollectorBinaries(sdk2.Resource):
    releasable = True
    releasers = INVENTORI_RELEASERS
    executable = True
    ttl = GENERAL_TTL
    arcadia_build_path = 'inventori/bin/timeseries/realty/realty'
    task_name = 'history_timeseries_realty'
    task_executable_resource_description = (
        'Inventori generate realty timeseries executable binaries'
    )

    @register_inventori_task(
        legacy_key='INVENTORI_REALTY_HISTORY_COLLECTOR_BINARIES',
        name='history_timeseries_realty',
        path_to_task_class='inventori.pylibs.tasks.timeseries.realty_history'
                           '.RealtyHistoryDataCollector',
    )
    class InventoriTaskParams(InventoriBaseTaskParams):
        result_folder = parameters.StrParam(
            'Result folder',
            details='(Default //home/inventori/[prod|test]/timeseries/history_data/realty)',
            default=create_inventori_default_path_getter('timeseries/history_data/realty'),
        )


class InventoriVideoHistoryCollectorBinaries(sdk2.Resource):
    releasable = True
    releasers = INVENTORI_RELEASERS
    executable = True
    ttl = GENERAL_TTL
    arcadia_build_path = 'inventori/bin/timeseries/video/video'
    task_name = 'history_timeseries_video'
    task_executable_resource_description = (
        'Inventori generate video timeseries executable binaries'
    )

    @register_inventori_task(
        legacy_key='INVENTORI_VIDEO_HISTORY_COLLECTOR_BINARIES',
        name='history_timeseries_video',
        path_to_task_class='inventori.pylibs.tasks.timeseries.make_video_history.VideoHistoryDataCollector',
    )
    class InventoriTaskParams(InventoriBaseTaskParams):
        result_folder = parameters.StrParam(
            'Result folder',
            details='(Default //home/inventori/[prod|test]/timeseries/history_data/video)',
            default=create_inventori_default_path_getter('timeseries/history_data/video'),
        )

        pi_data_folder = parameters.StrParam(
            'Pi data folder',
            details='(Default //home/inventori/[prod|test]/daily_pi_data/chronology)',
            default=create_inventori_default_path_getter('daily_pi_data/chronology'),
        )


class InventoriDailyPiDataBinary(sdk2.Resource):
    releasable = True
    releasers = INVENTORI_RELEASERS
    executable = True
    ttl = GENERAL_TTL
    arcadia_build_path = 'inventori/bin/etl/pi_data_collector/pi_data_collector'
    task_name = 'daily_pi_data_collector'
    task_executable_resource_description = (
        'Inventori pi_data_collector executable binaries'
    )

    @register_inventori_task(
        legacy_key='INVENTORI_DAILY_PI_DATA_BINARY',
        name='daily_pi_data_collector',
        path_to_task_class='inventori.pylibs.tasks.etl.pi_data_collector.PiDataColletor',
    )
    class InventoriTaskParams(InventoriBaseTaskParams):
        result_folder = parameters.StrParam(
            'Result folder',
            details='(Default //home/inventori/[prod|test]/daily_pi_data)',
            default=create_inventori_default_path_getter('daily_pi_data'),
        )


class InventoriDailyBudgetRecommendationBinary(sdk2.Resource):
    releasable = True
    releasers = INVENTORI_RELEASERS
    executable = True
    ttl = GENERAL_TTL
    arcadia_build_path = 'inventori/bin/recommendations/daily_budget_recommendation/daily_budget_recommendation'
    task_name = 'daily_budget_recommendation'
    task_executable_resource_description = (
        'Inventori daily_budget_recommendation executable binaries'
    )

    @register_inventori_task(
        legacy_key='INVENTORI_DAILY_BUDGET_RECOMMENDATION_BINARY',
        name='daily_budget_recommendation',
        path_to_task_class='inventori.pylibs.tasks.recommendations.make_daily_budget_rec'
                           '.DailyBudgetRecommendation',
    )
    class InventoriTaskParams(InventoriBaseTaskParams):
        result_folder = parameters.StrParam(
            'Result folder',
            details='(Default //home/inventori/[prod|test]/recommendations/daily_budget_recommendation)',
            default=create_inventori_default_path_getter('recommendations/daily_budget_recommendation'),
        )


class InventoriDailyCampaignsStatCollector(sdk2.Resource):
    releasable = True
    releasers = INVENTORI_RELEASERS
    executable = True
    ttl = GENERAL_TTL
    arcadia_build_path = 'inventori/bin/etl/daily_campaigns_stat/daily_campaigns_stat'
    task_name = 'prepare_campaigns_daily_stat'
    task_executable_resource_description = (
        'Inventori generate daily_campaigns_stat executable binaries'
    )

    @register_inventori_task(
        legacy_key='INVENTORI_DAILY_CAMPAIGNS_STAT_COLLECTOR',
        name='prepare_campaigns_daily_stat',
        path_to_task_class='inventori.pylibs.tasks.etl.daily_campaigns_stat'
                           '.PrepareCampaignsDailyStat',
    )
    class InventoriTaskParams(InventoriBaseTaskParams):
        result_folder = parameters.StrParam(
            'Result folder',
            details='(Default //home/inventori/[prod|test]/stat)',
            default=create_inventori_default_path_getter('stat/daily_campaigns_stat'),
        )


class InventoriMainPagePredictionSlicerBinary(sdk2.Resource):
    releasable = True
    releasers = INVENTORI_RELEASERS
    executable = True
    ttl = GENERAL_TTL
    arcadia_build_path = (
        'inventori/bin/main_page_prediction_slicer/main_page_prediction_slicer'
    )
    task_name = 'main_page_prediction_slicer'
    task_executable_resource_description = (
        'Inventori generates main_page_prediction_slicer executable binaries'
    )

    @register_inventori_task(
        legacy_key='INVENTORI_MAIN_PAGE_PREDICTION_SLICER_BINARY',
        name='main_page_prediction_slicer',
        path_to_task_class='inventori.pylibs.tasks.main_page_prediction_slicer'
                           '.main_page_prediction_slicer.MainPagePredictionSlicer',
    )
    class InventoriTaskParams(InventoriBaseTaskParams):
        result_dir_path = parameters.StrParam(
            'Result folder',
            details='(Default //home/inventori/[prod|test]/main_page/predictions)',
            default=create_inventori_default_path_getter('/main_page/predictions'),
        )

        inventori_server_url = parameters.StrParam(
            'Inventori Server',
            details='Inventori server for predictions',
            default='ogdiwxzvzipd7bie.vla.yp-c.yandex.net',
        )


class InventoriMainPagePredictionScorerBinary(sdk2.Resource):
    releasable = True
    releasers = INVENTORI_RELEASERS
    executable = True
    ttl = GENERAL_TTL
    arcadia_build_path = (
        'inventori/bin/main_page_prediction_scorer/main_page_prediction_scorer'
    )
    task_name = 'main_page_prediction_scorer'
    task_executable_resource_description = (
        'Inventori generates main_page_prediction_scorer executable binaries'
    )

    @register_inventori_task(
        legacy_key='INVENTORI_MAIN_PAGE_PREDICTION_SCORER_BINARY',
        name='main_page_prediction_scorer',
        path_to_task_class='inventori.pylibs.tasks.main_page_prediction_scorer'
                           '.main_page_prediction_scorer.MainPagePredictionScorer',
    )
    class InventoriTaskParams(InventoriBaseTaskParams):
        result_dir_path = parameters.StrParam(
            'Result folder',
            details='(Default //home/inventori/kravchenkoegv/[prod|test]/main_page_predictions_scorer)',
            default=create_inventori_default_path_getter('main_page_predictions_scorer',
                                                         home='//home/inventori/kravchenkoegv'),
        )

        inventori_server_url = parameters.StrParam(
            'Inventori Server',
            details='Inventori server for predictions',
            default='ogdiwxzvzipd7bie.vla.yp-c.yandex.net',
        )


class InventoriServerLogsParserBinary(sdk2.Resource):
    releasable = True
    releasers = INVENTORI_RELEASERS
    executable = True
    ttl = GENERAL_TTL
    arcadia_build_path = (
        'inventori/bin/server_logs_parser/server_logs_parser'
    )
    task_name = 'server_logs_parser'
    task_executable_resource_description = (
        'Inventori generates server_logs_parser executable binaries'
    )

    @register_inventori_task(
        legacy_key='INVENTORI_SERVER_LOGS_PARSER_BINARY',
        name='server_logs_parser',
        path_to_task_class='inventori.pylibs.tasks.server_logs_parser.server_logs_parser'
                           '.ServerLogsParser',
    )
    class InventoriTaskParams(InventoriBaseTaskParams):
        original_log_folder = parameters.StrParam(
            'Папка, где лежат сырые логи',
            details='(Default //home/inventori/logs/deploy-logs)',
            default='//home/inventori/logs/deploy-logs',
        )
        intermediate_log_folder = parameters.StrParam(
            'Папка, где будут лежать промежуточные логи',
            details='(Default //home/inventori/logs/daily_parsed_deploy-logs)',
            default='//home/inventori/logs/daily_parsed_deploy-logs',
        )
        expanded_log_folder = parameters.StrParam(
            'Папка, где будут лежать распаршенные логи ивентов',
            details='(Default //home/inventori/logs/inventori_events_logs)',
            default='//home/inventori/logs/inventori_events_logs',
        )
        start_date = parameters.StrParam(
            'Обрабатывать начиная с этой даты',
            details='(Default 2022-03-01)',
            default='2022-03-01',
        )
        force = parameters.BoolParam(
            'Force',
            details='Если True, то будут обработаны все даты начиная с start_date',
            default=False,
        )


@register_binary_inventori_task(custom_runner=True)
class InventoriPerformanceEtlBinary(sdk2.Resource):
    releasable = True
    releasers = INVENTORI_RELEASERS
    any_arch = True
    auto_backup = True
    ttl = GENERAL_TTL
    arcadia_build_path = (
        'inventori/bin/etl/performance_etl_collector/performance_etl_collector'
    )
    task_name = 'performance_etl_collector'
    task_executable_resource_description = (
        'Inventori performance etl executable binaries'
    )

    class InventoriTaskParams(InventoriBaseTaskParams):
        # force_validation = parameters.BoolParam('Force database validation', default=True)

        yt_database_dir = parameters.StrParam(
            'YT database dir',
            details='Path to store database results ',
            default=create_inventori_default_path_getter(
                relative_path=lambda p: 'performance/bases/' + datetime.now().strftime('%Y-%m-%d'),
            ),
        )

        ttl = parameters.IntParam(
            'YT ttl time for copied recommendation actual folder',
            default=30,
            required=True
        )


class InventoriRotateAndSupportHelperBinaries(sdk2.Resource):
    releasable = True
    releasers = INVENTORI_RELEASERS
    any_arch = True
    auto_backup = True
    ttl = GENERAL_TTL
    arcadia_build_path = (
        'inventori/pylibs/inventori_duty/duty_rotate_and_support_helper/bin/duty_rotate_and_support_helper'
    )
    task_name = 'duty_rotate_and_support_helper'
    task_executable_resource_description = (
        'Inventori duty rotate and support helper'
    )

    @register_inventori_task(
        legacy_key='INVENTORI_ROTATE_AND_SUPPORT_HELPER_BINARIES',
        name='duty_rotate_and_support_helper',
        path_to_task_class='inventori.pylibs.inventori_duty.duty_rotate_and_support_helper'
                           '.duty_rotate_and_support_helper.InventoriDutyRotateAndTicketCreatorHelper',
    )
    class InventoriTaskParams(InventoriBaseTaskParams):
        startrek_token = parameters.YavSecretParam(
            'Startrek token secret id',
            default_yav_secret=ROBOT_INVENTORI_SECRET,
            default_yav_secret_key='startrek_token',
        )

        juggler_token = parameters.YavSecretParam(
            'Juggler token',
            default_yav_secret=ROBOT_INVENTORI_SECRET,
            default_yav_secret_key='juggler_token',
        )

        ttl = parameters.IntParam(
            'YT ttl time for copied recommendation actual folder',
            default=30,
            required=True
        )


class InventoriPerformanceDatabase(sdk2.Resource):
    releasable = True
    releasers = INVENTORI_RELEASERS
    any_arch = True
    auto_backup = True
    ttl = GENERAL_TTL


class InventoriPrepareAutoruDBBinary(sdk2.Resource):
    releasable = True
    releasers = INVENTORI_RELEASERS
    executable = True
    ttl = GENERAL_TTL
    arcadia_build_path = 'inventori/bin/etl/autoru/prepare_autoru_db/prepare_autoru_db'
    task_name = 'prepare_autoru_db'
    task_executable_resource_description = (
        'Inventori prepare_autoru_db executable binaries'
    )

    @register_inventori_task(
        legacy_key='INVENTORI_PREPARE_AUTORU_DBBINARY',
        name='prepare_autoru_db',
        path_to_task_class='inventori.pylibs.tasks.etl.autoru.prepare_autoru_db.PrepareAutoruDB',
    )
    class InventoriTaskParams(InventoriBaseTaskParams):
        pass


class InventoriPrepareAutoruDspLogsBinary(sdk2.Resource):
    releasable = True
    releasers = INVENTORI_RELEASERS
    executable = True
    ttl = GENERAL_TTL
    arcadia_build_path = 'inventori/bin/etl/autoru/prepare_autoru_dsp_logs/prepare_autoru_dsp_logs'
    task_name = 'prepare_autoru_dsp_logs'
    task_executable_resource_description = (
        'Inventori prepare_autoru_dsp_logs executable binaries'
    )

    @register_inventori_task(
        legacy_key='INVENTORI_PREPARE_AUTORU_DSP_LOGS_BINARY',
        name='prepare_autoru_dsp_logs',
        path_to_task_class='inventori.pylibs.tasks.etl.autoru.prepare_autoru_dsp_logs'
                           '.PrepareAutoruDspLog',
    )
    class InventoriTaskParams(InventoriBaseTaskParams):
        pass


class InventoriPrepareRealtyDBBinary(sdk2.Resource):
    releasable = True
    releasers = INVENTORI_RELEASERS
    executable = True
    ttl = GENERAL_TTL
    arcadia_build_path = 'inventori/bin/etl/realty/prepare_realty_db/prepare_realty_db'
    task_name = 'prepare_realty_db'
    task_executable_resource_description = (
        'Inventori prepare_realty_db executable binaries'
    )

    @register_inventori_task(
        legacy_key='INVENTORI_PREPARE_REALTY_DBBINARY',
        name='prepare_realty_db',
        path_to_task_class='inventori.pylibs.tasks.etl.realty.prepare_realty_db.PrepareRealtyDB',
    )
    class InventoriTaskParams(InventoriBaseTaskParams):
        pass


class InventoriPrepareRealtyDspLogsBinary(sdk2.Resource):
    releasable = True
    releasers = INVENTORI_RELEASERS
    executable = True
    ttl = GENERAL_TTL
    arcadia_build_path = 'inventori/bin/etl/realty/prepare_realty_dsp_logs/prepare_realty_dsp_logs'
    task_name = 'prepare_realty_dsp_logs'
    task_executable_resource_description = (
        'Inventori prepare_realty_dsp_logs executable binaries'
    )

    @register_inventori_task(
        legacy_key='INVENTORI_PREPARE_REALTY_DSP_LOGS_BINARY',
        name='prepare_realty_dsp_logs',
        path_to_task_class='inventori.pylibs.tasks.etl.realty.prepare_realty_dsp_logs'
                           '.PrepareRealtyDspLogs',
    )
    class InventoriTaskParams(InventoriBaseTaskParams):
        pass


class InventoriAutoruProjectHistoryBinaries(sdk2.Resource):
    releasable = True
    releasers = INVENTORI_RELEASERS
    executable = True
    ttl = GENERAL_TTL
    arcadia_build_path = 'inventori/bin/history/autoru/autoru'
    task_name = 'prepare_autoru_history'
    task_executable_resource_description = (
        'Inventori generate prepare autoru history executable binaries'
    )

    @register_inventori_task(
        legacy_key='INVENTORI_AUTORU_PROJECT_HISTORY_BINARIES',
        name='prepare_autoru_history',
        path_to_task_class='inventori.pylibs.tasks.history'
                           '.autoru_media_project_history.PrepareAutoruMediaProjectHistory',
    )
    class InventoriTaskParams(InventoriBaseTaskParams):
        pass


class InventoriRealtyProjectHistoryBinaries(sdk2.Resource):
    releasable = True
    releasers = INVENTORI_RELEASERS
    executable = True
    ttl = GENERAL_TTL
    arcadia_build_path = 'inventori/bin/history/realty/realty'
    task_name = 'prepare_realty_history'
    task_executable_resource_description = (
        'Inventori generate prepare realty history executable binaries'
    )

    @register_inventori_task(
        legacy_key='INVENTORI_REALTY_PROJECT_HISTORY_BINARIES',
        name='prepare_realty_history',
        path_to_task_class='inventori.pylibs.tasks.history.realty_media_project_history'
                           '.PrepareRealtyMediaProjectHistory',
    )
    class InventoriTaskParams(InventoriBaseTaskParams):
        pass


@register_inventori_task(
    name='sandbox_guard',
    path_to_task_class='inventori.pylibs.tasks.sandbox_guard.SandboxGuard',
)
class InventoriSandboxGuardTaskParams(InventoriBaseTaskParams):
    sandbox_token = parameters.YavSecretParam(
        'Sandbox token',
        default_yav_secret=ROBOT_INVENTORI_SECRET,
        default_yav_secret_key='sandbox_token',
        env_arg='SANDBOX_TOKEN',
        required=True,
    )

    yt_dir = parameters.StrParam(
        'YT dir',
        details='Base cypress node directory for result tables to store',
        default='//home/inventori/schedulers',
    )

    now = parameters.DateTimeParam(
        'Exec datetime',
        details='now by default',
        default=parameters.DefaultGetter(lambda p: p.exec_time, require='exec_time'),
    )

    recreate_table = parameters.BoolParam(
        'Recreate daily table',
        skip_in=parameters.ParamSrc.SANDBOX_SDK2_PARAM,
    )


@register_inventori_task(
    name='direct_event_log_collector',
    path_to_task_class='inventori.pylibs.tasks.direct_event_log_collector.DirectEventLogCollector',
)
class InventoriDirectEventLogCollectorTaskParams(InventoriBaseTaskParams):

    # max_days_to_process = parameters.IntParam(
    #     'Max dates to process',
    #     default=20,
    # )

    start_date = parameters.StrParam(
        'Collect logs staring from this date',
        default='2022-03-01',
    )

    force = parameters.BoolParam(
        'Force',
        details='Forcefully update ignore output log freshness',
        default=False,
    )
