import datetime
import logging
import re

SOLOMON_PROJECT = 'inventori'
SOLOMON_SERVICE = 'scheduler'


class cached_property(object):
    def __init__(self, func):
        self.func = func

    def __get__(self, instance, cls=None):
        result = instance.__dict__[self.func.__name__] = self.func(instance)
        return result


def camel_case_to_snake_case(name):
    """
    >> camel_case_to_snake_case('HelloWorld1AndWorld2')
    'hello_world1_and_world2'

    >> camel_case_to_snake_case('GiveMeAPropertyID')
    'give_me_a_property_id'

    :type name: str
    :rtype: str
    """
    matches = re.finditer(r'.+?(?:(?<=[a-z\d])(?=[A-Z])|(?<=[A-Z])(?=[A-Z][a-z])|$)', name)
    return '_'.join(m.group().lower() for m in matches)


class class_property(object):
    def __init__(self, func):
        self.func = func

    def __get__(self, obj, type):
        return self.func(type)


def report_status(task, exec_timestamp, yt_cluster, status, logger=None):
    from inventori.pylibs.solomon import Solomon
    logger = logger or logging

    if not task.Parameters.send_status_to_solomon:
        logger.info('Status is %s (sending to solomon disabled)', status)
        return

    solomon_client = Solomon(
        oauth_token=str(task.Parameters.solomon_token.data()[task.Parameters.solomon_token.default_key]),
        project=SOLOMON_PROJECT,
        cluster=str(task.Parameters.environment_type),
        service=SOLOMON_SERVICE,
    )
    payload = [{
        'ts': exec_timestamp,
        'labels': {
            'yt_cluster': yt_cluster,
            'metric': 'status',
            'task_name': str(task.Parameters.task_name),
        },
        'value': status,
    }]

    if status == 1:
        payload.append({
            'ts': exec_timestamp,
            'labels': {
                'yt_cluster': yt_cluster,
                'metric': 'duration',
                'task_name': str(task.Parameters.task_name)
            },
            'value': int(datetime.datetime.now().timestamp()) - exec_timestamp,
        })

    logger.info('Sending %s status to solomon:\n%s', status, payload)
    solomon_client.send(payload)
