# -*- coding: utf-8 -*-

import logging
import sandbox.common.types.notification as ctn
from sandbox.projects.common.ya_deploy import release_integration
import sandbox.projects.inventori.artifacts as artifacts
import sandbox.projects.inventori.common.config as config
import sandbox.projects.inventori.common.consts as consts
import sandbox.projects.inventori.common.resources as resources
import sandbox.projects.inventori.common.yql_utils as yql_utils
import sandbox.projects.inventori.common.yt_utils as yt_utils
import sandbox.projects.inventori.etl.utils as utils
import sandbox.sandboxsdk.environments as sandbox_environments
import sandbox.sdk2 as sdk2
import subprocess
import sys

VALIDATION_ERROR_CODE = 99

LOGGER_NAME = 'inventori traffic generating'

logger = logging.getLogger(LOGGER_NAME)


class InventoriEtl(release_integration.ReleaseToYaDeployTask2, sdk2.Task):
    class Context(sdk2.Task.Context):
        has_new_resource = False

    class Requirements(sdk2.Requirements):

        environments = [
            sandbox_environments.PipEnvironment("yandex-yt"),
            sandbox_environments.PipEnvironment("yandex-yt-yson-bindings-skynet"),
            sandbox_environments.PipEnvironment('yandex-yt-yson-bindings-skynet', version='0.3.32-0'),
        ]
        disk_space = 80 * 1024  # 80 Gb for data (all necessary indices + archive)
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        release_new_resource, releaseTo = utils.releaseTo_params()
        ttl = sdk2.parameters.Integer("TTL for released resource (days, always; 0 for inf)", default=30, required=True)

        prepare_new_version_of_logs = sdk2.parameters.Bool(
            "Preparing new database (if not - download files from 'result' folder on YT)", default=True)

        force_validation = sdk2.parameters.Bool("Force database validation", default=True)

        yt_proxy = sdk2.parameters.String("YT server name", required=True)
        yt_token_vault_name = sdk2.parameters.String("YT key vault name")
        yt_token_vault_user = sdk2.parameters.String("YT key vault user")

        yql_proxy = sdk2.parameters.String("YQL server name", required=True)
        yql_token_vault_name = sdk2.parameters.String("YQL key vault name")
        yql_token_vault_user = sdk2.parameters.String("YQL key vault user")

        solomon_token_vault = sdk2.parameters.String('Solomon token',
                                                     description='Sandbox Vault secret name for Solomon token',
                                                     default='INVENTORI_OAUTH_TOKEN',
                                                     required=True)

        inventori_binary = sdk2.parameters.LastReleasedResource("inventori traffic binary resource", required=True)

    def get_general_env(self):

        yt_env = yt_utils.get_yt_env_from_parameters(self.Parameters)
        yql_env = yql_utils.get_yql_env_from_parameters(self.Parameters)

        final_env = dict(yt_env.items() + yql_env.items())

        yt_dir = config.TEST_YT_PROJECT_DIRECTORY if (
            self.Parameters.releaseTo != 'stable') else config.PROD_YT_PROJECT_DIRECTORY
        final_env['YT_DIR'] = yt_dir

        final_env['FORCE_VALIDATION'] = '1' if self.Parameters.force_validation else '0'

        final_env['INVENTORI_SOLOMON_TOKEN'] = sdk2.Vault.data(self.Parameters.solomon_token_vault)

        return final_env

    def get_yp_oauth_token(self):
        return sdk2.Vault.data("INVENTORI_YP_TOKEN")

    def on_execute(self):

        env = self.get_general_env()

        inventori_binary_data = sdk2.ResourceData(self.Parameters.inventori_binary)
        inventori_binary_data_path = str(inventori_binary_data.path.joinpath(artifacts.INVENTORI_ETL))

        if self.Parameters.prepare_new_version_of_logs:

            try:

                with sdk2.helpers.ProcessLog(self, logger=logging.getLogger(LOGGER_NAME)) as pl:
                    subprocess.check_call(
                        [inventori_binary_data_path],
                        stdout=pl.stdout,
                        stderr=pl.stderr,
                        env=env
                    )

            except subprocess.CalledProcessError as ex:
                if ex.returncode == VALIDATION_ERROR_CODE:
                    self.server.notification(
                        subject="VALIDATION FAIL for task #{task_id}".format(task_id=self.id),
                        recipients=["inventori-monitoring@yandex-team.ru"],
                        transport=ctn.Transport.EMAIL,
                        type=ctn.Type.HTML,
                        charset=ctn.Charset.UTF,
                        task_id=self.id,
                        view=ctn.View.EXECUTION_REPORT,
                    )
                    sys.stderr.write("VALIDATION FAIL")

                raise

        yt_data_folder = "%s/%s" % (env['YT_DIR'], "result")
        archive_created = utils.read_tables_and_create_archive(logger=logger,
                                                               log_processor=lambda: sdk2.helpers.ProcessLog(self, logger=LOGGER_NAME),
                                                               yt_proxy=self.Parameters.yt_proxy,
                                                               yt_token=env['YT_TOKEN'],
                                                               yt_data_folder=yt_data_folder,
                                                               csv_tables_prefix_to_columns=consts.CSV_FORMAT_TABLES_PREFIX_TO_COLUMNS,
                                                               json_tables_prefix_to_columns=consts.JSON_FORMAT_TABLES_PREFIX_TO_COLUMNS)

        if archive_created:
            resource = sdk2.ResourceData(
                resources.InventoriDatabase(
                    self,
                    "Inventori database in tar.gz",
                    consts.PLATFORM_YT_ARCHIVE_NAME
                )
            )

            resource.ready()
            self.Context.has_new_resource = True

    def on_success(self, prev_status):
        utils.send_email_and_deploy(self, prev_status)
